rng = Random.default_rng()
Random.seed!(99)
#Constants
C = 0.01


#Initial Conditions
I = [1.0, 0.0]   #Psi(0)=1, Psi'(0)=1
etaspan = (0.05, 5.325)

#radius range
datasize= 100
etasteps = range(etaspan[1], etaspan[2]; length = datasize)


function whitedwarf(du, u, p, r)
    psi = u[1]
    dpsi = u[2]
    du[1] = dpsi
    du[2] = (-((psi^2-C))^(3/2) - 2/r * dpsi)
end


#Defining the Ordinary differential equation as an ODEProblem with the DifferentialEquations.jl
prob = ODEProblem(whitedwarf, I, etaspan)
#Solving the ODEProblem with the Tsit5() algorithm
sol = solve(prob,saveat=etasteps)

dudt2 = Lux.Chain(Lux.Dense(2, 80, tanh),Lux.Dense(80, 80, tanh), Lux.Dense(80, 2))
#Setting up the NN parameters randomly using the rng instance
p, st = Lux.setup(rng, dudt2)

etasteps =  etasteps[1:end-10]
etaspan = (etasteps[1], etasteps[end])
I=[1.0,0]
prob_neuralode = NeuralODE(dudt2, etaspan, Tsit5(); saveat = etasteps)

function predict_neuralode(p)
    Array(prob_neuralode(I, p, st)[1])
end
#Training data
true_data= Array(sol[:,1:end-10])
### Define loss function as the difference between actual ground truth data and Neural ODE prediction
function loss_neuralode(p)
    pred = predict_neuralode(p)
    loss = sum(abs2, true_data .- pred)
    return loss, pred
end

p_trained = (layer_1 = (weight = Float32[-0.5836619 0.4376292; -0.8438847 -0.63573873; -0.22818492 -0.16087975; -0.058397036 -1.2134186; -0.66212684 0.6330059; -0.6018261 -0.83270854; 0.4901169 -0.39823735; 0.33831456 -0.53536886; -1.1017618 0.24213287; 0.15590152 1.5958364; -0.8491539 -0.049485054; 0.58421326 -0.62068045; 0.27540046 -0.024816215; 0.1388686 -1.1266612; -0.3256207 0.067228585; 0.8663603 -0.36634585; 0.5500781 -0.07733675; 0.64877224 -0.83104366; 0.4919355 0.7158439; 0.06852026 0.68308914; -0.34694472 0.12416278; 0.6570016 -0.0932242; -0.87470794 -0.16549127; 0.5981263 -0.123304054; 0.7964552 -0.15019432; 0.06314678 0.80510193; -0.641391 -0.0006132077; 0.16636837 1.1728938; -0.07892441 -0.11649985; 0.9746405 0.67424834; 0.4482575 0.6243109; -0.7161666 -0.21974187; 0.7785012 -0.21918012; -0.89574087 0.743669; -0.4264137 -1.4958879; 0.6460656 -0.52937335; -0.60968894 0.12614328; 0.30978617 1.3631048; 0.6447419 0.14467306; -0.7083478 -1.5734134; -0.5764454 0.3449271; -0.7420932 0.031896327; -0.115757786 -0.68838435; -0.7848534 0.044804778; -0.47406423 1.1070065; 0.60586345 -0.4251131; -0.92915577 0.7042489; -0.43897098 -0.6338367; 0.40875518 -0.52939004; 0.36264953 -0.11258128; -1.0538148 0.36766702; 0.8031039 -0.24348474; -0.4483093 -0.930856; 0.72940296 0.13522026; -0.5192829 -0.06552727; -0.40019205 0.005540917; 0.8011548 0.04203193; -0.8637946 0.23198454; -0.9509497 0.12896462; 0.5303764 1.2414685; 0.47005555 0.76475227; -0.630275 0.26209795; 0.5185801 -0.57950395; -0.29712626 0.697611; -0.54864514 -0.696941; 0.7915787 0.35139608; 1.0143443 -0.82775736; 1.0244046 0.2786547; -1.032274 0.51786345; 0.8747835 0.53835523; 1.071358 -0.24084942; 0.77858096 -0.67578053; -0.7707391 0.20725952; -0.21004583 -0.5400134; -0.07694214 -1.5469654; 0.54887563 1.4507291; -0.27408078 0.26656792; 0.43058622 0.951756; -0.5737836 0.09376381; -0.41529992 -0.6823074], bias = Float32[-0.7261357; -0.3940102; 0.43703058; -0.31599912; -0.69152457; -0.23458612; 0.89062893; 0.9270689; -0.81647927; 0.018953865; -0.74108577; 0.6475313; -0.21473932; -0.2290055; -0.965899; 0.8197715; 0.83668983; 0.514139; 0.18303564; -0.5864642; 0.56615776; 0.042237546; -0.9466446; 0.83810353; 0.93466127; -0.20967546; -0.92141837; -0.025643019; -0.53786623; 0.8833189; -0.02521705; -0.8925286; 0.69938093; -0.67027867; -0.37881997; 0.7953787; -0.93308395; -0.21105576; 0.74155873; -0.39467135; -0.59104; -0.89519167; 0.30798772; -0.9387116; -0.06764954; 0.82132787; -0.81434244; 0.184269; 0.76518285; 0.22247554; -0.7661458; 0.86382854; -0.123442285; 0.9453286; -0.09463076; -0.72832555; 0.97155446; -0.78622156; -0.83431476; 0.47445178; 0.68112254; -0.8429541; 0.6456532; -0.44352978; -0.68279034; 0.67084366; 0.39983174; 1.3204399; -0.6879069; 0.18245637; 0.71518546; 0.7950578; -0.8961277; 0.24540922; 0.20833266; 0.110288486; -0.87119687; -0.05110301; -0.8877154; -0.35762513]), layer_2 = (weight = Float32[-0.79266834 0.8982822 0.683286 -0.19177446 -0.84237576 0.92468953 0.66386455 -0.7834701 0.9440418 0.5413672 -0.13743044 0.6847691 -0.7845979 0.34739718 0.51500857 -0.6161726 0.12717547 0.41288126 0.748182 0.7763306 -0.10911831 -0.047705993 -0.16911782 0.5151288 -0.95107514 0.79908144 -0.28731057 0.46907642 -0.3214008 -0.9232973 0.84681875 0.5083256 0.4989304 0.027047165 0.20248392 0.8733633 -0.28274122 0.44324556 0.21719253 -0.03016438 -0.3515198 0.23113358 0.6652752 0.4115307 -0.14274842 0.6502272 0.16248849 -0.8009326 0.8883013 -0.33159807 0.67132527 0.63457555 -0.79855406 -0.43863147 0.8381644 -0.27044213 -0.65777683 -0.02416389 0.6667091 -0.5083465 -0.34066948 -0.7424268 0.17013638 0.778118 0.17777726 -0.9555539 -0.8249071 -0.7423909 0.90688854 -0.66421086 -0.8876557 -0.37349677 1.0497975 0.6997214 -0.25254595 0.044331294 0.34971964 0.79002184 -0.6227599 -0.5657342; -0.3763834 0.40887463 0.46208796 -0.74317646 -0.6352258 0.549208 0.382233 -0.21098301 0.46133983 0.7378652 0.5847139 0.07707151 -0.48809594 -0.6912047 -0.6812638 -0.48196942 0.7072162 -0.76654917 0.520398 0.46249506 0.7602266 -0.518851 0.67542243 0.47778824 -0.71824193 0.5809745 -0.5317201 0.525267 -0.41171235 -0.65009284 0.40553316 -0.55376667 0.5419663 0.71300155 -0.7026964 0.37700957 -0.40525272 0.5907165 -0.81901973 -0.7995879 0.84624726 -0.3653502 0.2920658 -0.4310295 0.65888494 0.27782825 0.7536615 -0.38581234 0.44727796 -0.6909712 0.4711024 0.44308278 -0.41371167 0.70334464 0.49249333 -0.4718861 -0.8123032 -0.6742238 -0.28218645 0.736466 0.5152123 -0.59636647 -0.9988611 0.7163541 -0.7132917 -0.67797554 -0.50070995 -0.5631543 0.8331012 -0.69570255 -0.71487653 -0.69238955 0.699226 0.40176663 -0.57995856 0.623589 -0.5458172 0.67747915 -0.6487877 -0.6820929; -0.25924093 0.44254187 -0.20773046 0.7178975 -0.3574947 0.44895536 0.5065821 -0.44979784 0.2512776 -0.71435684 -0.6263768 0.45250192 0.29216358 0.53889567 0.4710979 -0.49200404 -0.89754 0.5074357 0.3938783 0.25049683 -0.8406282 0.66218984 -0.6353346 -0.79521996 -0.52814436 0.39861614 0.6654418 -0.62931496 0.6137355 -0.3028369 0.27163753 0.53318316 -0.73683286 -0.84966254 0.5036116 0.15694344 0.7441524 -0.84178823 0.6899143 0.71683174 -0.50412107 0.82909846 0.46550882 0.60777634 -0.69135493 0.24658892 -0.86789316 -0.3216717 0.20548122 0.59256375 0.2425316 -0.28713483 -0.25307223 -0.8444345 -0.26909974 0.74086785 -0.06209254 0.8235056 0.5084485 -0.7187826 -0.5639001 0.28713748 0.75172156 0.22086458 0.6167889 -0.28810012 -0.38854307 0.6967916 0.1164651 -0.47555956 -0.29578543 0.53721094 0.5466392 0.36502042 0.67906934 -0.67384464 0.5773839 0.24062392 0.28775942 0.28417775; 0.35908863 0.064978816 -0.30077252 0.91085035 0.011510708 0.38050392 -0.40212044 -0.6033204 -0.15099578 -0.6776714 -0.8678908 0.3201534 0.61625063 0.71180016 0.5895463 -0.41612366 -0.81335974 0.6688695 0.01683032 0.37275267 -0.82699424 0.84014624 -0.83400226 -0.92182285 -0.5567657 -0.29612342 1.0584775 -0.54851913 0.8299802 0.033435136 0.36615536 0.77505875 -0.6229533 -0.9076923 0.8057765 -0.35489976 1.0534741 -0.594249 0.61535966 0.83859575 -0.74451584 1.0095638 -0.05768032 1.116828 -0.7792205 0.11396727 -0.53614265 -0.11284376 -0.06454041 0.6967713 0.01840929 -0.7086907 0.17184418 -1.0503287 -0.37298745 0.9195025 0.38077816 0.8976505 0.4532196 -0.8925928 -1.09238 0.8012004 0.73167264 0.10099526 0.9710734 0.23315543 0.014558195 0.36871836 -0.16480902 -0.2783111 -0.18771192 0.79620796 0.0008546052 -0.15133215 0.89238787 -0.9102442 0.6529669 -0.2856712 0.704878 0.56956565; 0.20013534 -0.15423256 0.27247137 -0.9626821 0.17761098 -0.44117957 -0.28085917 0.461495 -0.22172977 0.820093 0.78077644 -0.60436195 -0.16854914 -0.70638597 -0.71839094 0.30255437 0.8463073 -0.59080696 -0.42145753 -0.396951 0.8532398 -0.6491329 0.78171515 0.9755145 0.4542573 0.04771711 -0.7560016 0.5812428 -0.65018904 0.23533826 -0.47489366 -0.74214554 0.8388968 0.8317599 -0.90797347 -0.09374936 -0.9122831 0.74830073 -0.8819443 -0.7705304 0.69857186 -0.72200805 -0.4709571 -0.8022739 0.84287804 -0.40485814 0.97219235 0.43274754 -0.17779559 -0.84889513 0.053797744 0.51040405 0.30908665 0.7805656 0.22588198 -0.6430882 -0.031658765 -0.66512 -0.40233493 0.6760746 0.6316233 -0.2955824 -0.6885016 -0.110262774 -0.7846608 0.12319472 0.30629608 -0.6734405 0.10472512 0.21279922 0.37561917 -0.80405027 -0.1697255 -0.055192098 -0.8351311 0.8997864 -0.5148004 -0.13036884 -0.50796396 -0.27179092; 0.94196564 -0.9340294 -0.84187037 0.17404401 0.6238631 -0.56625605 -0.5929464 0.7527138 -0.741584 -0.41610792 0.19498047 -0.48921818 0.64426726 -0.32311186 -0.390994 0.8752898 -0.09669651 -0.24759994 -0.5774405 -0.9719702 0.47048935 -0.11208471 -0.054402422 -0.2426355 0.7965469 -1.0475551 -0.063069716 -0.7228623 0.44536564 1.0779786 -0.77259356 -0.41242722 -0.33816785 0.09610638 -0.49917257 -0.63307273 -0.05949957 -0.7499939 -0.31770036 0.010149363 0.2533542 -0.22306342 -0.87748796 -0.39564538 0.053912893 -0.9006164 -0.16885227 0.66439986 -0.68652284 0.3322747 -1.0466176 -0.61707246 0.7433787 0.49938577 -0.71104664 0.39389816 0.67698073 -0.120500885 -0.90127724 0.4936587 0.124144375 0.51858103 -0.3299439 -0.8465107 -0.31350836 0.9220722 1.0652468 0.87468094 -0.9165216 0.7139456 0.74714506 0.067357115 -0.80288947 -0.92005163 0.20572479 -0.034096163 -0.32987112 -0.9381263 0.48818278 0.6491199; 0.811286 -0.74256754 -0.7892444 -0.5820828 0.58511126 -0.7600566 -0.7265882 0.66259277 -0.55379915 0.21894342 0.5856006 -0.85354954 0.6816785 -0.43959087 -0.45537075 0.43536445 0.11004968 -0.5387672 -0.6857896 -0.6240857 0.5585973 -0.29888153 0.49156398 0.20377412 0.6832422 -0.52151394 -0.43340102 -0.35052675 -0.223797 0.5665197 -0.83818305 -0.40762383 -0.21502842 0.3935712 -0.5212256 -0.8136149 -0.10352612 -0.3567737 -0.57438916 -0.15875474 0.69163483 -0.3705757 -0.60708266 -0.2378014 0.46050853 -0.8214027 0.35010716 0.59350634 -0.6490314 -0.4413597 -0.7043104 -0.6986497 0.86881036 0.25951925 -0.84096074 -0.42056647 0.7314141 -0.14228892 -0.42741683 0.4125611 0.25631258 1.0863637 -0.58367145 -0.70696694 -0.5265089 0.61097836 0.53941953 0.68959844 -0.578714 0.41993323 0.44764116 -0.3048788 -0.6185719 -0.88058096 -0.2136968 0.4609817 -0.5900485 -0.73886025 0.96111405 0.7773359; -0.7023651 0.49394253 0.7458105 0.44090766 -0.6758221 0.6447407 0.516143 -0.43357453 0.52835023 -0.44830114 -0.59450203 0.32749316 -0.8577107 0.5220305 0.4027598 -0.5296477 -0.58061665 0.6115194 0.5583484 0.83544016 -0.2821237 0.3778768 -0.32502356 -0.22639498 -0.80807483 0.6576907 0.5061975 0.021612734 0.2395777 -0.7271966 0.5035506 0.5946509 -0.15493673 -0.5899216 0.44916284 0.802457 0.6118462 -0.16432025 0.46230853 0.40618154 -0.26452556 0.32933602 0.69464964 0.63748455 -0.4927383 0.8411063 -0.5341919 -0.66666466 0.808009 0.2624096 0.56536674 -0.10188518 -0.490251 -0.27342218 0.86398214 0.37450007 -0.5584335 0.3729137 0.39406177 -0.44586796 -0.6379334 -0.8280496 0.325416 0.5837375 0.52101064 -0.8321372 -0.8374249 -0.97054315 0.74416316 -0.57779455 -0.7413316 0.58724207 0.535506 0.7574253 0.5400291 -0.53891486 0.50583977 0.5577157 -0.61470217 -0.7277825; -0.69013155 0.6851956 0.60152334 -0.4662764 -0.75684714 0.9386737 0.60133153 -0.89574254 0.5795417 0.5824998 0.60448027 0.79037297 -0.68672913 -0.5928849 0.0015920943 -0.73748434 0.60168767 -0.4460639 0.7743621 0.6185754 0.31886724 -0.5496878 0.5976711 0.5702348 -0.8912196 0.6637803 -0.37593764 0.74694234 -0.46139714 -0.5761355 0.6011916 -0.42505828 0.38749284 0.43028513 -0.26442856 0.515993 -0.4429383 0.6873748 -0.29610327 -0.40380988 0.3951082 -0.30541506 0.69289035 -0.43790692 0.4363431 0.52048856 0.76421875 -0.65884674 0.68058187 -0.6266958 0.74704134 0.6695314 -0.537459 0.16575515 0.5913043 -0.72692156 -0.79954094 -0.5356883 0.8281281 0.28448987 0.36017385 -0.59101504 -0.35658026 0.7350999 -0.47391072 -0.5642985 -0.5083944 -0.4548245 0.6858987 -0.76707953 -0.6736578 -0.41604 0.61721325 0.5617577 -0.67083347 0.6255261 0.7194963 0.44077802 -0.62549025 -0.56105745; -0.2196951 0.01624216 0.67255324 -0.8703183 0.050184976 -0.18564636 0.05223671 0.2930583 0.25625837 0.723773 0.8941515 -0.4726175 -0.46976474 -0.8297759 -0.8788901 0.36002493 0.64795125 -0.7911068 0.02356706 -0.29865053 0.60493124 -0.9406727 0.785228 0.60944736 0.3932692 0.07425666 -0.6340549 0.6161636 -0.8520034 -0.25199133 -0.19508696 -0.93714994 0.5352234 0.7019591 -0.6281292 0.15568079 -0.7832622 0.59852403 -0.73504823 -0.9447867 0.9120872 -0.89256865 0.047008496 -0.9084992 0.7913351 0.05176017 0.6829796 0.3090305 0.079087034 -0.90267676 0.06789711 0.8315652 -0.16467576 0.77036583 0.6493144 -0.86409384 -0.4159052 -0.95872974 -0.28050378 0.60781986 0.97546446 -0.6553016 -0.88219887 0.110334314 -0.7753005 -0.21496114 -0.19787855 -0.5358593 0.38786265 0.14128807 -0.015883535 -0.9698468 0.080409326 0.05397161 -0.76182985 0.76248825 -0.29341957 0.005772044 -0.6691953 -0.6369765; -0.26466626 0.36245248 -0.29148108 0.9433843 -0.50527096 0.4704472 0.1918772 -0.5258969 0.34119195 -0.96299857 -0.75215083 0.62618494 0.5604044 0.7291894 0.61703634 -0.22684382 -0.69902116 0.83934265 0.19175534 0.3822871 -0.68606466 0.85731727 -0.7712952 -0.6665836 -0.379114 -0.019804168 0.6726047 -0.72771597 0.7808259 -0.38094008 0.55170053 0.7703092 -0.53510714 -0.7060089 0.68807447 0.32422346 0.7807493 -0.6327151 0.60528165 0.9419818 -0.7548445 0.637923 0.52026874 0.757219 -0.9246327 0.49005532 -1.016428 -0.48857853 0.3127737 0.8453722 0.019239273 -0.31618932 -0.1934141 -0.8522117 -0.12485898 0.90730786 0.17717202 0.7775401 0.502048 -0.79240686 -0.868564 0.42434442 0.7908836 0.3793596 0.8162232 -0.26694077 -0.15636766 0.66341823 -0.12090296 -0.51561975 -0.31926635 1.0234987 0.34052458 0.14929762 0.71752787 -0.8988897 0.40499368 0.14280751 0.37380427 0.49118367; 0.2892747 -0.46737224 0.16436888 -0.8315492 0.28715774 -0.574884 -0.3872333 0.37975153 -0.2844913 0.66909575 0.6777241 -0.41041768 -0.35505378 -0.5931622 -0.84223735 0.3705241 0.93713665 -0.61062104 -0.34926897 -0.36502942 0.6924644 -0.8242485 0.81753504 0.7721409 0.38604134 -0.365332 -0.693535 0.47963288 -0.85104626 0.34390733 -0.28911284 -0.61900795 0.7136197 0.62878865 -0.6771863 -0.18043762 -0.8271187 0.7500733 -0.8737414 -0.82810414 0.8061372 -0.80618787 -0.40839514 -0.5966192 0.72947603 -0.21234207 0.8261844 0.5589399 -0.23209944 -0.85536104 -0.35160875 0.36455053 0.45832944 0.57702875 0.35441813 -0.85591 -0.17015477 -0.7283898 -0.38465837 0.5189828 0.6858227 -0.3287294 -0.8793633 -0.3230516 -0.5553597 -0.004121194 0.11389598 -0.45606568 0.12611033 0.44255662 0.5136608 -0.9030792 -0.35717967 -0.45515728 -0.7410973 0.6954147 -0.81117445 -0.16995977 -0.31619757 -0.282597; -0.61492544 0.9889612 0.6498457 -0.38123617 -0.6826282 0.7843728 0.6726313 -0.7270475 0.9797481 0.42443752 0.12188233 0.65409774 -0.8225079 0.04685782 0.3010811 -0.835087 0.3594543 -0.023052132 0.87174904 0.8649264 0.058062054 -0.26636067 0.4047648 0.64746 -0.82818484 0.90711516 -0.42121205 0.7533909 -0.46203566 -0.965601 0.9092148 0.16269329 0.692278 0.4240036 -0.105243996 0.6775032 -0.51007247 0.6385853 -0.017102508 -0.34013325 -0.18705153 0.004730537 0.9597867 -0.054126363 0.50592613 0.93936616 0.5648028 -0.96982014 0.58148044 -0.52110577 0.820134 0.47269106 -0.935561 -0.37320927 0.7676068 -0.64592737 -0.57928026 -0.19329631 0.67929864 -0.10103881 0.22427084 -0.65753806 0.118274 0.8688041 -0.1287761 -0.6761837 -0.68779105 -0.8206171 0.6637202 -0.61300343 -0.847489 -0.45477527 0.99996275 0.9015937 -0.5494333 0.28316605 0.5519238 0.7408272 -0.45934936 -0.66084576; -0.8468455 0.64078623 0.7017411 0.44326437 -0.70924485 0.81550646 0.4737477 -0.65844166 0.8525565 0.21587706 -0.27927738 0.6624102 -1.0283786 0.3219314 0.61154836 -0.94299626 -0.46438017 0.4522007 0.51013595 0.6286205 -0.32427984 0.43835044 -0.4523267 -0.15359469 -0.6784529 0.694572 0.50516945 0.5031376 0.123904444 -0.87807184 0.66479653 0.6993852 0.07457677 -0.08084472 0.48430818 0.8074476 0.43316513 0.3598636 0.1530251 0.39512032 -0.21042518 0.61292386 0.59869266 0.44050416 -0.18721765 0.4048067 -0.014355707 -0.62196934 0.7015597 0.13815734 1.0731317 0.16935234 -0.844235 -0.51659024 0.99078196 0.24032854 -0.9076517 0.575895 0.43292996 -0.6614148 -0.48068753 -0.6709525 0.41460326 0.68895227 0.5183467 -0.8046122 -0.9027399 -1.0349087 1.0269976 -0.59892935 -0.7063209 0.03733563 0.89739895 0.73167294 0.11140863 -0.27554727 0.62811637 0.8107952 -0.63014096 -0.90069234; 0.583483 -0.59346414 -0.6813817 0.5444203 0.5649822 -0.3876102 -0.60358435 -0.23412897 -0.45174596 -0.7529917 -0.6104601 -0.3815427 0.54413646 0.64089346 1.1335897 0.06831734 -0.8647148 0.65135694 -0.5382214 -0.38909543 -0.6338751 0.7265133 -0.40029436 -0.5630699 0.34191945 -0.36500785 0.69581693 -0.6987222 0.55976003 0.5921772 -0.36332652 0.9479782 -0.69961256 -0.4831591 0.7929173 -0.65642023 0.80363804 -0.532287 0.5154224 0.48910677 -0.5152607 0.66258824 -0.52189916 0.65502435 -0.6377927 -0.6216071 -0.56568944 0.34719044 -0.5534708 0.6383347 -0.55903053 -0.8981216 0.494518 -1.0268514 -0.47187117 0.65942186 0.57639027 0.91956776 0.48696715 -0.9539545 -0.8081477 0.7873998 0.680825 -0.47614154 0.765844 0.65354306 0.36214834 0.5924034 -0.35838214 0.58206457 0.5051659 0.6691017 -0.5285287 -0.6021822 0.7984824 -0.62018675 0.4549945 -0.7455733 0.81174356 0.6949888; 0.6732382 -0.6685349 -0.72670853 0.34160468 0.65508133 -0.6925833 -0.58924615 0.81861985 -0.88355315 -0.7160209 -0.08638156 -0.47890088 0.4974601 -0.34623834 -0.29576567 0.810288 -0.4346195 -0.3891182 -0.6220149 -0.81665254 0.23489553 -0.12843247 0.11900723 -0.58480287 0.7064961 -0.6891112 0.2417201 -0.723794 0.2989454 0.96480316 -0.555351 -0.47450885 -0.44800085 -0.008987202 -0.33278412 -0.64114165 0.17602505 -0.6096524 -0.015185045 -0.0025414794 0.06942211 -0.1379035 -0.7821809 -0.09835046 -0.19093846 -0.64733094 -0.23722734 0.88973993 -0.61305404 0.24296328 -0.7188553 -0.5668381 0.9813974 0.60161513 -0.81634873 0.2303616 0.9921268 -0.0006330296 -0.60560596 0.27249092 0.10108495 0.6100441 -0.43367395 -0.78953147 -0.30607757 0.83437574 0.9299542 0.7653012 -0.838822 0.7761148 1.0774109 0.36489624 -1.0209408 -0.9330097 0.451595 -0.37478414 -0.64408845 -0.8931787 0.414842 0.44484556; 1.0455412 -0.84677285 -0.7177231 0.19784254 1.1600443 -0.8466829 -0.9707171 0.43273714 -0.4817224 -0.3311336 0.04873681 -0.75193715 0.45116112 -0.2940231 -0.4003189 0.5294363 -0.4037322 -0.51032066 -0.7921952 -0.75271624 0.023067938 0.2016008 0.028498877 -0.7780155 0.74591494 -0.6306497 0.620734 -0.78129435 0.6296565 0.5850034 -0.9645166 -0.04313887 -0.8115802 -0.22209239 0.18492016 -1.1028601 0.6976107 -0.45140716 -0.30989087 0.21619755 0.37390456 0.37079787 -0.9659068 0.31505463 0.09713636 -1.0158317 -0.09191991 0.95824623 -1.1294426 0.28100893 -0.57963 -0.8424699 0.8001475 0.058300134 -0.68499875 0.5458146 0.7293016 0.2247581 -0.54514223 0.22370231 0.014690899 0.6392968 -0.299851 -0.8273204 -0.004360346 0.7233222 0.7866265 0.39864185 -0.64697224 0.8305168 0.63095653 0.18555993 -0.75123554 -0.7090816 0.43928754 -0.1550224 -0.6203837 -0.82662886 0.8964424 0.653167; 0.9042608 -0.73368734 -0.76849973 -0.35052145 1.0698844 -0.74781525 -1.0642163 0.44775748 -0.48588082 -0.346188 0.41733357 -0.91716325 0.6979627 -0.5465576 -0.5529806 0.5565041 -0.11509145 -0.6554241 -0.90777653 -0.7492656 0.24538255 -0.22913705 0.43055004 -0.30197978 0.5655685 -0.8356814 0.25244123 -0.8108222 0.32037354 0.66332513 -0.971979 -0.13361968 -0.7391683 0.35062346 -0.36949527 -0.95112467 0.31047532 -0.57325494 -0.4621355 -0.22872826 0.43356523 -0.1142434 -0.9442651 -0.004865409 0.42564023 -0.98437774 0.14775553 0.70570517 -1.0492177 -0.24546306 -0.675839 -0.7593787 0.84433734 0.25330123 -0.70538884 0.3543665 0.541537 -0.23236713 -0.5209759 0.10706409 0.15848528 0.8270579 -0.68512213 -0.6235545 -0.19883513 0.74205744 0.5516247 0.6381529 -0.79480416 0.93179864 0.65459716 -0.27382222 -0.51194614 -0.8325302 0.4346059 0.022805855 -0.44633487 -0.75654566 0.7528324 0.72264457; 0.5120606 -0.8180693 -0.58707154 0.4170513 0.6618278 -0.7103227 -0.6215334 0.70211977 -0.8573516 -0.4193191 -0.42860514 -0.90287167 0.4987267 0.48523116 -0.38097373 0.80546093 -0.56714404 0.23989762 -0.5899129 -0.7009211 -0.3039531 0.67509675 -0.44462138 -0.5801357 0.959771 -0.6293586 0.42479625 -0.4194888 0.46322063 0.6288905 -0.88970923 -0.012811836 -0.72356886 -0.33004847 0.5169535 -0.64807427 0.71064276 -0.6116849 0.4650579 0.5945314 -0.29525992 0.35196903 -0.74596804 0.28433082 -0.5798918 -0.5283981 -0.3814331 0.9196462 -0.7389609 0.48846203 -0.7602187 -0.4110869 0.6236794 -0.11718674 -0.46419492 0.60699517 0.5537438 0.32094273 -0.825315 -0.44702846 -0.5114494 0.46730548 0.13635007 -0.5564784 0.2817334 0.47415054 0.7806633 0.66634476 -0.47438824 0.8171659 0.74277943 0.74315476 -0.7544435 -0.6160638 0.3702301 -0.48115942 -0.29899907 -0.5685902 0.46495214 0.582189; -0.8036079 0.76664156 0.7992904 -0.5478456 -0.6917779 0.81435853 0.7884773 -0.7046415 0.91458654 0.53820115 0.6024371 0.5932802 -0.66152114 -0.30727604 0.25936818 -0.8413539 0.3844233 -0.3185413 0.5801563 0.9140667 0.044629913 -0.38425717 0.2777968 0.6243712 -0.9892429 0.5779891 -0.49670935 0.3870821 -0.51140195 -0.6156997 0.7416675 0.28098327 0.60519713 0.32522485 -0.118935354 0.6114781 -0.47303912 0.56969064 -0.36664987 -0.5320119 0.2960991 -0.14276673 0.79531014 -0.16634718 0.41044682 0.7603904 0.6143773 -0.7320292 0.72055244 -0.33398923 0.74663126 0.5154569 -0.85711545 -0.20497832 0.78253883 -0.46478114 -0.82577217 -0.5276512 0.6726149 0.13234264 0.11968044 -0.37483534 -0.07735666 0.61588854 -0.19764447 -0.8464609 -0.5605511 -0.49175823 0.74811906 -0.9206521 -0.8039684 -0.68959945 0.98410475 0.663525 -0.67753625 0.33364737 0.6781982 0.5386823 -0.5718905 -0.49339175; 0.9463456 -0.7223752 -0.84855026 0.033164512 0.89319146 -0.66165066 -0.7060642 0.6275402 -1.0145019 -0.25807634 0.0149327945 -0.56576 0.7079392 -0.33309355 -0.6643271 0.9355382 -0.100419 -0.37065485 -0.72748303 -0.81904405 0.32461652 0.046652928 0.029570749 -0.30387914 0.7260841 -0.8184996 -0.185716 -0.7137052 0.20818032 0.7513645 -0.5881545 -0.5451849 -0.19098644 -0.046679396 -0.5145769 -0.75634253 -0.14689125 -0.72550374 -0.25845316 -0.08813596 0.38395903 -0.4379335 -0.6456758 -0.42706928 0.12157967 -0.78683406 -0.21610878 0.686593 -0.6477113 0.19682021 -0.9170514 -0.59810966 0.66164446 0.59552175 -0.6879645 0.08934134 0.95717084 -0.26170135 -0.5544073 0.40420112 0.4587566 0.73230386 -0.47738424 -0.8594932 -0.22428763 0.92944884 1.1038104 0.56906825 -0.918422 0.66353846 1.0524297 0.104432985 -0.9793887 -0.6170559 0.060114376 -0.12268306 -0.45821166 -1.0497321 0.66443807 0.67853665; -0.31400374 0.4305817 0.3710929 0.6781595 -0.33767802 0.30347735 0.5361468 -0.3589043 0.33504608 -0.6025694 -0.69384515 0.44417793 -0.35511616 0.48415452 0.6079306 -0.47546452 -0.71564394 0.7053056 0.6111511 0.28505933 -0.6110548 0.7019005 -0.58763474 -0.9560376 -0.61209357 0.30769095 0.60732204 -0.89589566 0.5853002 -0.56505686 0.51973426 0.5845369 -0.6750932 -0.56523925 0.85184556 0.49641994 0.67992216 -0.8884185 0.7917727 0.8317553 -0.7999158 0.5345581 0.3214347 0.6316747 -0.615658 0.54348344 -0.6504297 -0.25035924 0.5575557 0.7608823 0.48347527 -0.85109985 -0.3571201 -0.73764884 0.34184256 0.6720033 -0.5241499 0.64732635 0.5146836 -0.5263377 -0.689862 -0.18106279 0.4382061 0.289097 0.6457539 -0.45155802 -0.52211016 0.24431266 0.39329237 -0.37030947 -0.34349418 0.7951158 0.2842098 0.3596981 0.7618338 -0.73452264 0.49540588 0.37632814 0.03938209 -0.25848567; -0.08356756 0.25068647 -0.32936105 0.91358805 -0.31649196 0.49832296 0.25245622 -0.42776638 0.17767134 -0.63435996 -0.95431715 0.33138525 0.31219327 0.7217854 0.7140267 -0.23653954 -0.8987279 0.77760816 0.52074367 0.35751587 -0.5753355 0.8821826 -0.62169003 -0.9311541 -0.20810446 0.109670654 0.7537199 -0.53292876 0.9350506 -0.15276314 0.34917316 0.5084466 -0.92047966 -0.9941854 0.6770081 0.3056041 0.9760262 -0.6971261 0.5598263 0.75136894 -0.67233646 0.65755564 0.36050245 0.8890996 -0.6417076 0.25121924 -0.83536696 -0.5733078 0.44018707 0.76224303 0.23727047 -0.5600499 -0.122189224 -0.63132 -0.38092855 0.8713118 0.12334928 0.69293195 0.39232734 -0.8155618 -0.64938164 0.32523212 0.7409564 0.36179674 0.5821901 -0.1509598 -0.22511645 0.38866118 -0.1760459 -0.3730804 -0.15975702 0.9070344 0.10433924 0.31087005 0.95668983 -0.8709715 0.48426437 -0.014914554 0.47303465 0.22470804; -0.86453557 0.98295534 0.76909333 -0.2581812 -0.73014927 0.7908697 0.70127857 -0.60573274 0.88925683 0.6676843 -0.21196838 0.77172035 -0.6465453 0.12728117 0.57838297 -0.58352244 0.19301859 0.4242965 0.8066082 0.94928646 -0.39434436 -0.14705685 -0.14997032 0.24205555 -0.92301327 0.9314395 -0.048953958 0.4728482 -0.32919616 -1.0232748 0.64210886 0.5448168 0.55933756 0.21037328 0.10937427 0.7095682 -0.20278955 0.37561315 0.23926985 0.09827665 -0.13272347 0.12974484 0.8527365 0.12782268 0.05243136 0.93223786 0.3666324 -0.9476652 0.68020445 -0.2711604 0.88598233 0.3150831 -0.7421914 -0.20230475 0.71285063 -0.2620132 -0.87032646 -0.08952162 0.805954 -0.48117614 -0.0075240433 -0.46595275 0.48379555 0.8769132 0.21270205 -0.8457802 -1.0133582 -0.7340771 0.75480634 -1.0103914 -1.0010093 -0.36680767 0.94555604 0.8057947 -0.4305935 0.17017888 0.5496006 0.74338484 -0.46338868 -0.55745286; 0.23118395 -0.44654053 0.46223596 -0.7972561 0.39362818 -0.566355 -0.102646105 0.29812163 -0.40231922 0.95066476 0.64315736 -0.47740066 -0.41051647 -0.8286725 -0.64255595 0.42432308 0.63518435 -0.7051117 -0.3878762 -0.5524308 0.84132 -0.59253645 0.7583447 0.9756146 0.2638783 -0.05495263 -0.66230243 0.5972967 -0.93464833 0.41489664 -0.36523858 -0.78076905 0.70824033 0.8948426 -0.7784773 -0.26458982 -0.6874622 0.6714911 -0.781441 -0.7224284 0.6403324 -0.8859585 -0.21208724 -0.7994145 0.87771827 -0.5307292 0.9328032 0.32238957 -0.1344084 -0.62701434 -0.04743467 0.44411424 0.4076689 0.6605583 0.34690237 -0.6274391 0.03449642 -0.845163 -0.55337757 0.866181 0.7582565 -0.13253713 -0.64769894 -0.14669698 -0.64384556 0.08031259 0.069650285 -0.8352336 -0.087034985 0.44253913 0.51804036 -0.8681287 -0.4720085 -0.16690232 -0.95558727 0.9509483 -0.27773824 -0.14581451 -0.41332164 -0.13744134; 0.2073991 0.21620646 -0.3172306 0.86385125 -0.0835698 0.4601628 0.15122534 -0.37795368 -0.037457295 -0.5962545 -0.8858346 0.5383507 0.38585913 0.6766142 0.50590277 -0.4897446 -0.73875856 0.84485 0.26770023 0.29334956 -0.8637549 0.8382014 -0.9855304 -0.81756455 -0.27017102 -0.2256078 0.9473482 -0.7946937 0.7498716 -0.07541802 0.46159294 0.5618097 -0.6725021 -0.8126394 0.68001413 0.20647486 0.91005486 -0.5956362 0.574506 0.66226965 -0.88948995 0.62196755 0.078083105 0.8560298 -0.7837485 0.2011103 -0.80728924 -0.415487 0.023907764 0.7455452 0.02031553 -0.6564651 -0.2758447 -0.54248667 -0.41023672 0.950448 0.48118868 0.6678797 0.2970436 -0.551431 -0.6100829 0.35610256 0.7988299 -0.038817007 0.75648695 0.11791756 -0.15035929 0.6413749 -0.1482543 -0.3328052 -0.031025458 0.91609466 0.087925576 0.13867949 0.8175306 -0.7238217 0.31437927 0.13948478 0.30797222 0.35588473; -0.8167405 0.8633277 1.0182542 0.1994808 -1.0010271 0.8107934 0.9781185 -0.5119655 0.78815866 0.033507746 -0.38999397 0.7308417 -0.83268756 0.53002 0.30938143 -0.6895711 -0.051891916 0.69431436 0.8226984 0.52948976 -0.40669253 0.39943516 -0.34238243 0.17448312 -0.48206043 0.6092284 0.12811251 0.69767237 0.040049173 -0.77554846 0.770823 0.5028714 0.29623866 -0.44443345 0.43343493 0.9473727 0.15068789 0.2766725 0.59920424 0.3603902 -0.63439703 0.3190146 0.96016175 0.114010036 -0.24023062 0.85095775 -0.24381647 -0.6382211 0.7799682 0.24149747 0.8665979 0.67522746 -0.8970999 -0.5305208 0.8397996 0.1310727 -0.6707967 0.16027097 0.495304 -0.2145161 -0.46383363 -1.01522 0.4149168 0.6125271 0.48437995 -0.86918646 -0.63898224 -0.4779567 0.74525464 -0.8246952 -0.6154232 0.2275346 0.7020598 0.9794944 0.18653679 -0.27415547 0.43375146 0.84719193 -0.98129135 -0.6775317; 0.67462564 -0.5914272 -0.5901282 0.5867969 0.72980183 -0.48920873 -0.6858564 -0.21054988 -0.32225493 -0.62991804 -0.5287608 -0.3043474 0.65937424 0.46775833 0.9060448 0.42405608 -0.90314674 0.6798618 -0.8185868 -0.61228126 -0.75081414 0.56043726 -0.626051 -0.76863337 0.30620548 -0.32119125 0.90198874 -0.5942357 0.6733926 0.3414164 -0.30137506 0.9255351 -0.6436619 -0.3637977 1.0144371 -0.7629672 0.89840907 -0.5015002 0.4708326 0.8636604 -0.42355138 0.76769066 -0.7196667 1.0161718 -0.39157286 -0.8135747 -0.46879774 0.6787948 -0.7607217 0.6819538 -0.2157263 -0.6860336 0.7687318 -1.1499625 -0.489841 0.5726094 0.58094144 0.8776142 0.5525787 -0.8957526 -0.78453916 0.79747725 0.8003523 -0.5728772 0.90881133 0.31993172 0.5636584 0.29669645 -0.5797961 0.49463147 0.3156737 0.46517748 -0.4194719 -0.58847755 0.90478367 -0.59603584 0.572253 -0.48180538 0.6079689 0.6344739; 0.071724206 0.088906415 -0.55646276 0.7704257 -0.3002049 0.20575467 -0.116210334 -0.31159347 -0.09219774 -0.5333049 -0.7078772 0.25385433 0.3771218 0.7393395 0.8329547 -0.39216468 -0.74813944 0.69358087 0.27914825 0.3888224 -0.6747312 0.6671681 -0.73544425 -0.64297634 -0.20972142 -0.18790163 1.0003954 -0.43146366 0.7725391 0.0077699893 0.22105439 0.6995126 -0.77247715 -1.0312299 0.8524248 0.054428317 0.64423287 -0.78259987 0.9579165 0.69624454 -0.668584 1.004833 -0.031186834 0.9341358 -0.8051467 0.19465396 -0.7562795 -0.3976218 -0.0934909 1.0024669 -0.054945357 -0.53610766 -0.1174696 -0.70549595 -0.60221094 0.7083162 0.36021334 0.9528258 0.39821628 -0.65370035 -0.6306891 0.53194857 0.58382714 -0.12762775 0.6735038 0.13248725 -0.045141872 0.69821537 -0.30500174 0.106686525 -0.3024011 0.8979239 -0.12824233 -0.10636666 0.7493677 -0.98013836 0.42816252 0.07464679 0.70147264 0.46108145; 0.13358185 -0.00993287 -0.35780254 0.80490035 -0.020046175 0.24292909 0.007448801 -0.20396252 -0.2054027 -0.58709264 -0.9148848 0.37173018 0.47651634 0.74970335 0.8419168 -0.3507905 -0.9347393 0.6673021 0.14787231 0.15266119 -0.8135492 0.7792544 -0.7678244 -0.91242754 -0.35447508 -0.06654083 0.91001177 -0.6081724 0.7636977 -0.11667863 0.18850869 0.90128976 -0.62240255 -1.0220634 0.7774392 -0.30261457 0.7895823 -0.48543054 0.71884304 0.7429817 -0.8592336 0.8231822 -0.0513855 0.9613681 -0.9769212 0.22322394 -0.62434006 -0.25738308 0.022224013 0.84246296 -0.19587961 -0.4776633 0.074170865 -0.59716195 -0.3926755 0.77373433 0.5603546 0.65196687 0.638518 -0.6361705 -0.8609037 0.5683014 0.82293415 -0.2359172 0.9810779 0.3299566 0.061455745 0.4582074 -0.2284231 -0.13168861 -0.10274632 0.7805261 -0.040445685 -0.18527023 0.7556924 -0.72872216 0.33355114 -0.28013694 0.40346953 0.4460827; -0.6600918 0.772782 0.89625067 0.32719386 -0.81456757 0.7173408 0.8108354 -0.50014216 1.0429386 0.35299042 -0.1519625 0.74118525 -0.63549817 0.3409754 0.27225175 -0.8822215 -0.16083959 0.32311633 0.75689256 0.6928899 -0.15050593 0.059339184 -0.3418931 0.24993749 -0.6927896 0.9477706 0.120019846 0.34041086 0.090475805 -0.9377535 0.5477889 0.63592744 0.28648555 0.0043523638 0.6313268 0.80658585 -0.10000556 0.43999067 0.3914339 0.5434679 -0.4637615 0.3611057 0.97762084 0.57293177 -0.31168607 0.59826034 0.13980098 -0.6095293 0.8387351 0.04981923 1.0319656 0.4997787 -0.62896657 -0.35765332 0.7151248 -0.18064669 -1.0597855 0.086658634 0.8996926 -0.3116911 -0.62066954 -0.5552845 0.42686597 0.71340704 0.47720084 -0.8590703 -0.9764754 -0.80126715 0.95621943 -0.81130296 -0.6571552 0.21604536 0.6699685 1.0322913 0.17151737 0.014643361 0.41020235 0.739768 -0.7326234 -0.8403505; -0.15435101 -0.4034486 0.30610833 -0.7464967 0.3764299 -0.33790705 -0.0014493039 0.49246714 0.07069468 0.80946505 0.83862084 -0.3189473 -0.4664499 -0.74923545 -0.5556918 0.45563114 0.7435471 -0.859474 -0.07469085 -0.48153967 0.6654985 -0.8001589 0.80972546 0.70576453 0.20543866 -0.037117712 -0.7931514 0.60550106 -0.764544 0.30548254 -0.41199502 -0.5976546 0.6104589 0.8246211 -0.96361136 -0.05267668 -0.8326803 0.6169723 -0.8991504 -1.0417285 0.8831995 -0.7177807 -0.18113835 -0.82635564 0.8000111 -0.12677613 0.93557906 0.17071183 -0.24286601 -0.9678587 0.2591973 0.39724883 -0.06252305 0.87024695 0.40930107 -0.9345082 -0.27377614 -0.9394729 -0.5668488 0.7253626 0.7329821 -0.47724336 -0.81406254 0.0053326692 -0.754666 0.062778115 -0.22146091 -0.50712997 0.21073632 0.21207103 0.06748136 -0.94012374 -0.33919656 -0.114239134 -0.6950721 0.9430432 -0.30104846 -0.08055567 -0.33431128 -0.53965443; 0.63984215 -0.49902537 -0.47315392 -0.82115006 0.5890641 -0.33385023 -0.596982 0.26489502 -0.3781224 0.6856328 0.692167 -0.671523 0.43326744 -0.6395111 -0.3579941 0.5285258 0.43477884 -0.6585809 -0.40628228 -0.2569516 0.7373568 -0.68999004 0.65492064 0.44457102 0.46147057 -0.5594479 -0.52124244 0.72022086 -0.7435265 0.37911615 -0.62364876 -0.6962611 0.47077653 0.6257547 -0.5784841 -0.43233672 -0.545102 0.81020373 -0.78864795 -0.66050065 0.5494879 -0.60603654 -0.666469 -0.50458586 0.89593065 -0.3671729 0.919806 0.6532105 -0.4121116 -0.5052548 -0.22589147 0.20330115 0.38065043 0.45467472 -0.27382383 -0.54884654 0.119232155 -0.5196585 -0.24410571 0.6420077 0.49993858 0.4522242 -0.9084234 -0.40237 -0.7284901 0.26024577 0.34006283 -0.5821087 -0.4028614 0.61674994 0.45029446 -0.93634754 -0.24522273 -0.5602707 -0.50431824 0.7634032 -0.27356678 -0.55742896 0.5092655 0.5328362; 1.0114776 -0.8514356 -0.9671909 -0.40791437 0.6915648 -0.8004603 -0.88793975 0.50724983 -0.92161554 -0.023283027 0.21566845 -0.86829656 0.93250877 -0.3325615 -0.4497931 0.6670963 0.31481823 -0.5281696 -0.81799203 -0.8676147 0.3762417 -0.3108407 0.5102973 -0.1975166 0.46543807 -0.6083945 -0.1909797 -0.34118778 -0.17693193 0.7277606 -0.5139756 -0.2805519 -0.30926284 0.2776916 -0.2845411 -0.8889604 -0.023288654 -0.50333625 -0.5043375 -0.48007104 0.37014362 -0.33616602 -0.7997878 -0.41757628 0.5154314 -0.833952 0.18043257 0.61308026 -0.9530101 -0.3788221 -0.58547217 -0.2673673 0.86683846 0.35365152 -0.99359274 -0.09785234 1.0039401 -0.34600556 -0.44779563 0.43231383 0.4841377 0.6391445 -0.25912738 -0.9202655 -0.46844408 0.9237293 0.65735286 0.78821415 -0.7471487 0.77713394 0.61963964 -0.14057536 -0.7318457 -0.91416395 -0.20715283 0.2069268 -0.5468635 -0.606864 0.6791757 0.67131186; 0.85149634 -0.86040634 -0.5410756 0.19669574 0.656114 -0.9500288 -0.88538814 0.5422854 -0.94765145 -0.4640081 -0.19195609 -0.64735264 0.4939231 0.09291637 -0.56648767 0.86324936 -0.24554998 -0.01005132 -0.65544075 -0.942984 -0.050244406 0.20314132 -0.29537347 -0.3280658 0.86665684 -0.63291746 0.31623012 -0.4432935 0.6009629 0.90498 -0.7186476 -0.34272233 -0.4321156 -0.43258193 0.05558549 -0.8939072 0.27735853 -0.53848237 -0.18081047 0.09600071 0.108344555 -0.062871516 -0.7872238 0.026731405 -0.45848045 -0.7645587 -0.46757635 0.6964161 -0.60201967 0.30294004 -0.7209322 -0.4141981 0.6552485 0.08675466 -0.71814287 0.36336368 0.7128171 0.27867404 -0.55270296 0.16238458 -0.13226546 0.4783351 -0.3504874 -0.647479 0.012694088 0.75642896 0.7172198 0.61738235 -0.6189792 0.72155774 0.9329303 0.48127005 -0.75684345 -0.72664803 0.27273902 -0.33177394 -0.32796702 -0.7942235 0.4357933 0.57811; -0.7925899 0.7927796 0.63056266 0.16980442 -0.7027303 0.5018388 0.52953213 -0.5269502 0.7458217 -0.042826563 -0.37809712 0.63915944 -0.6451228 0.19306488 0.33196208 -0.55411 -0.33836487 0.29542133 0.7089121 0.57173866 -0.61471856 0.4118784 -0.4536502 -0.23566109 -0.8491335 0.8320553 0.5591085 0.13782354 0.192186 -0.8259475 0.46575716 0.57453436 -0.3770037 -0.44120526 0.51551664 0.7830121 0.51828194 0.36705506 0.19648921 0.39103422 -0.5345014 0.3565331 0.71361125 0.47946274 -0.32898384 0.74370307 -0.1269718 -0.63012254 0.5913115 0.20746486 0.7225467 0.20912595 -0.7315975 -0.6207136 0.9673923 0.525453 -0.9287847 0.60126555 0.65511304 -0.40847635 -0.5723889 -0.8633506 0.5419534 0.6368719 0.54761696 -0.7490057 -0.9220472 -0.93919134 0.7035028 -0.7056862 -0.7257137 0.25150082 0.8564546 0.8496149 0.498043 -0.41263822 0.2924998 0.65796006 -0.68341166 -0.83465505; 0.71184146 -0.61793333 -0.5243012 -0.33583173 0.6015418 -0.4194757 -0.8036669 0.39440438 -0.8509831 0.37708536 0.45811883 -0.48909992 0.636025 -0.56770366 -0.5899665 0.5813214 0.3630433 -0.5988597 -0.4463542 -0.33589226 0.4311532 -0.4800419 0.39385125 0.4033686 0.5054614 -0.6724958 -0.35873285 0.46498522 -0.5524359 0.5300798 -0.59604114 -0.63001823 0.45641494 0.34939343 -0.58405054 -0.6320086 -0.43665975 0.27255052 -0.24677902 -0.52785134 0.35855895 -0.5696147 -0.67185074 -0.4868887 0.24415366 -0.3315257 0.4643923 0.3140118 -0.45420885 -0.3493304 -0.7474607 -0.4796157 0.5219393 0.30745465 -0.7401179 -0.49868894 0.69282645 -0.6355188 -0.43299246 0.58608794 0.5161005 0.45801297 -0.5587794 -0.8031674 -0.49164563 0.67950547 0.5890124 0.036819343 -0.5443082 0.5618409 0.8458446 -0.52715117 -0.5445356 -0.6518951 -0.41575035 0.6096243 -0.53658825 -0.738371 0.44081476 0.75124806; 0.39129135 -0.05899293 -0.37847242 0.97745556 -0.15995088 0.5212848 -0.03125535 -0.5471634 -0.21137337 -0.79913956 -0.658942 0.6292986 0.59056073 0.93242455 0.85202634 -0.34543553 -0.69709533 1.0193645 0.19541405 0.28009135 -1.0202806 0.8616527 -0.87499815 -0.594 -0.3842475 -0.4138006 0.8540118 -0.7206358 0.83771485 0.20731555 0.30764118 0.8148364 -0.55951536 -0.98820597 0.9660242 -0.20962931 0.5541332 -0.56936044 0.68203217 0.7827488 -1.0151461 0.8532325 -0.008905882 0.7362406 -0.902429 0.047554627 -1.0230671 -0.33869907 0.022836424 0.9464342 -0.34421018 -0.4862212 0.0058049075 -0.8369924 -0.63355625 0.5824318 0.41596487 0.6013119 0.5408795 -0.75929785 -0.79501927 0.6379506 0.70009583 -0.13407703 0.7872441 0.32029063 0.09698375 0.55780965 -0.3468496 -0.05841412 -0.12586361 0.942894 -0.31940496 -0.07550018 0.72267973 -0.6925804 0.54508585 -0.14031047 0.6566234 0.3864887; 0.7959594 -0.85487986 -0.5207178 -0.14047545 0.9860329 -0.54378396 -0.7644199 0.49198863 -0.85048413 -0.24423184 0.3424247 -0.6805607 0.78527784 -0.43474093 -0.3158752 0.4723681 0.036749244 -0.68147707 -0.79904205 -0.5898903 0.536022 -0.2815787 0.5694121 -0.19219927 0.46541488 -0.65614736 0.033785325 -0.64002305 -0.082082264 0.51023096 -0.77923626 -0.23835677 -0.42758125 0.2783474 -0.15563445 -1.0336562 0.23805113 -0.46135208 -0.39765593 -0.1362674 0.4869257 -0.26316944 -0.59246343 -0.23297386 0.48668665 -1.0310246 0.26259002 0.64023364 -0.7529499 -0.058842096 -0.691784 -0.94505894 0.7610424 0.15323448 -0.65217453 0.10256348 0.72012186 0.03728614 -0.4932108 0.22381635 0.3993018 0.68856597 -0.4984811 -0.5522842 -0.33967522 0.750477 0.65884036 0.50724596 -0.88347286 0.509195 0.659412 -0.29037902 -0.7636493 -0.6591434 0.08487606 -0.043139208 -0.23477517 -0.9112014 0.92773443 0.8539356; 0.7475261 -0.64462554 -0.6650235 0.48992595 0.5732821 -0.67319685 -0.50550485 0.89898556 -0.6040461 -0.7161162 -0.51262546 -0.64252084 0.74396133 0.2932988 -0.17247401 1.0399327 -0.42344853 0.48286414 -0.5330735 -0.7765725 -0.38880745 0.51158005 -0.6451713 -0.3370237 0.96754897 -0.71667737 0.5991583 -0.4931247 0.51672727 0.6341996 -0.78406227 0.24597856 -0.4193758 -0.63949776 0.19547024 -0.43044418 0.6891619 -0.45291832 0.37537432 0.39395583 -0.21461236 0.5697439 -0.536798 0.1927347 -0.6411608 -0.7080336 -0.59848726 0.6286178 -0.5410204 0.69492894 -0.6946321 -0.5677649 0.8548367 -0.016207939 -0.41142723 0.6057637 0.788355 0.2982024 -0.56917095 -0.2528672 -0.46388385 0.66208285 0.13664635 -0.7056829 0.27523753 0.7038825 0.8728726 0.5617236 -0.6081862 0.58125067 0.73363656 0.56749505 -0.89740115 -0.76275903 0.5899143 -0.5903362 -0.3098926 -0.7474735 0.69284785 0.5262483; 0.9511495 -0.56375915 -0.9124551 -0.5795384 1.0164537 -0.7486328 -0.9309125 0.42865112 -0.52171767 -0.0777026 0.54622066 -0.7733525 0.7553088 -0.6566086 -0.50950193 0.648203 0.067945726 -0.4330738 -0.9751138 -0.5067957 0.2916615 -0.33168876 0.4640982 -0.13810742 0.50866884 -0.78683674 0.019351894 -0.5672935 0.11899042 0.5031433 -0.63698196 -0.4237527 -0.43130162 0.5437907 -0.24939033 -1.0651013 -0.045465298 -0.5534298 -0.6263764 -0.43879965 0.58145297 -0.36953676 -0.78396034 -0.10265534 0.56249267 -0.6884608 0.46574926 0.68434584 -0.9245294 -0.13348436 -0.7257432 -0.75257504 0.7665872 0.49515143 -0.9579503 0.0053100386 0.80719876 -0.21043867 -0.63086605 0.40596825 0.29658931 1.0792273 -0.6823458 -0.5497281 -0.11296605 0.72899115 0.858454 0.7494137 -0.61705893 0.8419234 0.7103841 -0.16258986 -0.8148307 -1.0068591 -0.24320586 0.41990808 -0.34826425 -0.7179637 0.7342272 0.957842; 0.5593277 -0.06347121 -0.65813977 0.917863 0.10542155 0.2832114 -0.3809892 -0.6209215 -0.44179103 -0.61639863 -0.7369268 0.25180086 0.39626735 0.95690775 0.59727573 -0.18680519 -0.78312874 0.68872255 -0.042682335 0.256523 -0.69567037 0.71210927 -0.8313129 -0.77987236 -0.19089861 -0.32542244 0.92518264 -0.7043685 0.6196177 0.14603099 0.3023873 0.66431946 -0.5483303 -0.7447663 1.0150146 -0.4139745 0.8575969 -0.42876562 0.91372687 0.87124336 -0.9427988 0.74673873 -0.3356763 0.91719955 -0.7824297 -0.37479025 -0.76126665 -0.04237599 -0.1956712 0.7907766 -0.51674175 -0.54609615 0.46084395 -0.7077973 -0.6547703 0.66734606 0.27856398 0.8106799 0.33639908 -0.72517896 -0.9625961 0.5760889 0.63403755 -0.22485171 0.8659075 0.46743983 0.3911118 0.56214684 -0.32660452 0.35762277 -0.083439164 0.82789785 -0.21175581 -0.4484747 0.752509 -0.8710955 0.6986832 -0.46714482 0.5581655 0.7405837; -0.21242279 -0.1462722 0.2581954 -0.7478501 0.006325955 -0.37494257 0.13433512 0.60045904 0.14101553 0.6662216 0.7216954 -0.5377694 -0.48597974 -0.93210965 -0.61326444 0.6258673 1.0308485 -0.7510651 -0.29398003 -0.4071796 0.75522965 -0.7098003 0.7469642 0.64480674 0.29718718 0.34571669 -0.80278116 0.5423764 -0.8352453 -0.09304578 -0.37431225 -0.8571997 0.6645584 0.97962755 -0.75176316 -0.032947425 -1.0084524 0.50194186 -0.8678948 -0.7356789 0.8863575 -0.961237 -0.25385264 -0.7286287 0.8266846 -0.1667123 0.73133004 0.36326906 0.03972831 -0.9021192 0.28053847 0.7258299 -0.03187712 0.6445121 0.20535217 -0.75167924 -0.16679583 -0.8049542 -0.35041 0.89344066 0.72304046 -0.6704006 -0.7060844 0.0385168 -0.9212702 -0.07558827 -0.24826291 -0.5834289 0.07180928 0.2609647 0.044348698 -0.85608613 -0.24974068 -0.028456543 -0.7986789 0.7166989 -0.63156974 0.3232456 -0.46545875 -0.48451382; -0.6508319 0.95991075 0.5523548 0.060623307 -0.7708937 0.593472 0.65251815 -0.9346106 1.0547729 0.40861124 -0.29981902 0.6504321 -0.82470846 0.29413328 0.31825107 -0.8498979 0.2702245 0.24174199 0.79054475 0.948268 -0.40699217 0.2618267 0.10982736 0.39336532 -0.70772386 0.9066758 -0.1980385 0.32346293 -0.038111966 -0.88783276 0.611894 0.38091466 0.2701094 0.06385213 0.29669547 0.6362879 -0.15746906 0.7811418 0.42482942 0.23264347 -0.038690362 0.39094648 0.68390095 0.45848882 0.1490416 0.55440694 0.051561415 -0.71496725 0.68061316 -0.15004592 0.7758766 0.29307634 -0.7119277 -0.5830557 0.7238793 0.02785562 -1.0953872 0.040116623 0.7563117 -0.49955204 -0.16394912 -0.7226842 0.10512112 0.9593676 0.41721 -1.0492343 -1.1075009 -0.73369145 0.9973127 -0.7803654 -1.0336114 -0.3073105 1.0685084 0.86565286 -0.15921207 0.0118873445 0.56459004 0.8105981 -0.57743084 -0.67623; 0.5664196 -0.5067877 -0.12828308 -0.8358634 0.46380198 -0.50940925 -0.43044028 0.4353265 -0.27774677 0.60611415 0.48706943 -0.2855008 0.35959983 -0.7109096 -0.37072057 0.5345695 0.742748 -0.58182764 -0.3783246 -0.45293877 0.7031666 -0.79670364 0.8037644 0.6357337 0.42459998 -0.516385 -0.6780983 1.0606315 -0.7245683 0.49096826 -0.44435674 -0.76178575 0.7848988 0.7596044 -0.73920023 -0.47097296 -0.5718031 0.975625 -0.50731194 -0.58070356 0.5723391 -0.7372648 -0.37376562 -0.55156887 0.46535262 -0.45540187 0.74310195 0.40418535 -0.31563914 -0.580852 -0.51364994 0.36553347 0.39354727 0.61206317 -0.410106 -0.6910967 0.5751147 -0.79666203 -0.47115058 0.57747626 0.5593616 0.31867152 -0.54681927 -0.47562355 -0.51529014 0.31856 0.40411362 -0.38960114 -0.25850046 0.49124467 0.5504125 -0.4612069 -0.42867097 -0.51304394 -0.70279944 0.77789944 -0.3527773 -0.34791604 0.2062421 0.124022275; -1.0994841 0.7027967 0.71580845 0.13453004 -0.78218687 0.678765 0.7789672 -0.41250768 0.6860836 0.55622 -0.32432646 0.7768841 -0.6343217 0.4111676 0.5385153 -0.5263232 0.24205597 0.44764882 0.8243276 0.56460804 -0.30460262 0.37306312 -0.29811177 0.2824311 -0.6106961 0.68875486 -0.25212342 0.7111988 -0.0042743916 -0.7378902 0.8807121 0.4529781 0.8272426 -0.45853314 0.25728506 0.71170264 -0.19795911 0.57015866 0.4038285 0.070465356 -0.40060717 -0.16956437 0.58925825 0.37108257 -0.30184686 0.78465533 -0.38638213 -0.67871386 0.91931766 0.10596561 0.82541686 1.4020766 -0.6719228 -0.14271466 0.5536113 -0.045539424 -0.38430214 -0.2418265 0.461029 -0.22224072 -0.21544622 -0.87060666 0.590332 0.59924775 0.28124642 -0.7496785 -0.6304976 -0.18717334 0.55293274 -0.7008086 -0.6547574 0.30849427 0.58578354 0.96592754 -0.20813425 0.08830402 0.26999977 0.704241 -1.1708272 -0.9253619; 0.032033995 -0.23130207 0.29633516 -0.94401133 0.3472803 -0.53857577 -0.23406334 0.21202117 -0.14481266 0.9360933 0.8881505 -0.5001395 -0.5416392 -0.6090332 -0.7576122 0.4039583 0.6970242 -0.8542872 -0.09732229 -0.1972808 0.8542032 -0.76904863 0.8755753 0.68110925 0.31106457 -0.02902219 -0.92316705 0.48702 -0.91356397 0.06093066 -0.2226835 -0.66237605 0.768268 0.81678003 -0.8293173 -0.20298448 -0.7109203 0.76188475 -0.5310178 -1.0197934 0.8025978 -0.6681409 -0.35870454 -0.7547611 0.7885006 -0.31886053 1.0571582 0.25524285 -0.36155173 -0.81401455 -0.16434766 0.47815502 0.13355558 0.7913645 0.22055978 -0.86023086 -0.28857598 -0.519364 -0.57862055 0.92874736 0.6469641 -0.28717208 -0.67189324 -0.017070334 -0.87804765 -0.027147453 0.24381886 -0.40047646 0.14799435 0.2912258 0.20241146 -1.0551453 -0.14306961 -0.16169903 -0.70940816 0.9921241 -0.5780556 0.02700356 -0.47673014 -0.55269045; 0.70630914 -0.65010595 -0.4283715 0.6162818 0.522113 -0.9348025 -0.48811194 0.8767066 -0.8154212 -0.44362202 -0.5027716 -0.82895404 0.7391883 0.24740948 -0.19478011 0.8809234 -0.6080719 0.3478489 -0.6204482 -0.9754951 -0.4397212 0.41819984 -0.63972795 -0.7093081 0.7878106 -0.49282262 0.3486889 -0.6513327 0.64262754 0.85898006 -0.9291794 -0.043952655 -0.5491323 -0.6639361 0.26846796 -0.6864454 0.6984119 -0.71652204 0.42154425 0.5189607 -0.24207786 0.51989913 -0.8890284 0.51356107 -0.617194 -0.8914183 -0.7363173 0.83866984 -0.7011617 0.37982485 -0.79977757 -0.59181315 0.5541208 -0.044980526 -0.6456028 0.4277435 0.58631086 0.39188385 -0.9257572 -0.4332445 -0.2518119 0.4066828 0.069606505 -0.67955965 0.5016939 0.5433599 0.50885296 0.7962652 -0.8025885 0.5922116 0.7874738 0.51113224 -0.7339781 -0.5423273 0.7367082 -0.6876821 -0.6686571 -0.6541206 0.56744665 0.69885075; -0.2348898 0.30475962 -0.44865564 0.692746 -0.4902773 0.54583144 0.35847795 -0.47055733 0.341946 -0.9002528 -0.70032066 0.31991154 0.32678893 0.82802725 0.5931188 -0.3083072 -0.8045804 0.7979938 0.4781951 0.45561516 -0.84436244 0.77611107 -0.62802774 -0.75568104 -0.5459635 0.008960646 0.99171793 -0.60877067 0.89443856 -0.4956994 0.46795097 0.5447786 -0.8465909 -0.9135638 0.5762678 0.10905505 0.70787764 -0.83464134 0.75575054 0.8057039 -0.6974888 0.5612686 0.38797417 0.84895855 -0.6950861 0.31778288 -0.87106603 -0.5705377 0.21265873 0.8882927 0.22310828 -0.48696738 -0.11492998 -0.5366991 -0.3428418 0.81571454 -0.095269196 0.55178005 0.28314593 -0.5567485 -0.91599756 0.2807492 0.7023698 0.1193073 0.8040973 -0.25238138 -0.030525971 0.5213367 0.10933795 -0.15851727 -0.33424255 0.82154936 0.40291592 0.16300914 0.9569102 -0.92285377 0.5972206 0.18590535 0.45922458 0.49773088; 0.10435719 -0.29993978 0.5373337 -0.7726274 0.2956359 -0.5776967 -0.22229682 0.47815886 -0.22439764 0.7443517 0.88192296 -0.41491088 -0.57530415 -0.81655663 -0.47555307 0.2910485 0.7483516 -0.8056758 -0.35607868 -0.22795242 0.8091967 -0.976913 0.79839504 0.81138617 0.32029298 0.11837194 -0.7018809 0.68376166 -0.67479473 0.19691823 -0.45617422 -0.70479196 0.52623147 0.9496699 -0.90043956 -0.23282263 -0.95618176 0.74599904 -0.72199833 -1.0157442 0.60112685 -0.8218945 -0.38462213 -0.94569814 0.71079457 -0.38660786 0.8173425 0.20090462 -0.30545238 -0.914835 0.05152795 0.29179785 0.23818187 0.65884763 0.4436815 -0.81728405 -0.3507475 -0.7350694 -0.5465351 0.6328113 0.6880962 -0.3299556 -0.6545937 -0.21974246 -0.7319683 -0.103188105 0.0072429217 -0.5639114 0.30554494 0.31829304 0.16899806 -0.9508168 -0.055818092 -0.18760985 -0.6447502 1.0259162 -0.60896605 -0.03594519 -0.5924793 -0.34414327; -0.48789486 0.70460755 0.74648774 -0.66362625 -0.74486655 0.7437108 0.5637276 -0.63817656 0.6519757 0.75717765 0.23663764 0.5137117 -0.7442265 -0.13860486 0.20914936 -0.9371482 0.62812316 -0.06652252 0.6952395 0.8030498 0.10252612 -0.4234396 0.58028704 0.6562351 -0.8617345 0.82229865 -0.32255533 0.74870837 -0.34321678 -0.8472238 0.98008925 0.36360937 0.64331317 0.4816498 -0.17379124 0.7584145 -0.4756986 0.7808798 -0.09156229 -0.25307575 0.19935857 -0.026412142 0.5884318 -0.05973174 0.42432934 0.8263175 0.449028 -0.65922195 0.83190614 -0.57144797 0.8666383 0.5840866 -0.60319835 -0.3845826 0.55077064 -0.51360345 -0.6530432 -0.107710764 0.84189045 -0.04250525 0.25421178 -0.7224676 0.028613737 0.67078984 -0.26574656 -0.80631995 -0.5711212 -0.7969261 0.8509209 -0.9317696 -0.7001366 -0.59222555 0.75240326 0.85403067 -0.3953703 0.2712011 0.74385464 0.70044255 -0.38192323 -0.7460859; 0.540518 -0.22830701 -0.68253934 0.623638 0.25793114 0.104589164 -0.4311448 -0.18171872 -0.32586595 -0.7949468 -0.8819405 0.1685245 0.44470692 0.742292 0.75703907 -0.15778027 -0.5492111 0.8900956 -0.35824898 -0.055006117 -0.6438625 0.85973555 -0.64023894 -0.5668897 0.13531826 -0.4496242 0.8075297 -0.76937 0.7491625 0.29286492 0.033930108 0.7280299 -0.53126156 -0.695387 0.8506954 -0.38796738 0.57129925 -0.5320775 0.7610696 0.83269674 -0.7077516 0.72305155 -0.2219674 0.968521 -0.65371394 -0.57493776 -0.48851493 0.13635635 -0.34056672 0.6586312 -0.5335871 -0.5453544 0.43317986 -0.74842125 -0.52906954 0.623499 0.6078262 0.5972457 0.29063743 -0.6809989 -0.6976148 0.56679577 1.0090752 -0.39764223 0.85309476 0.37423855 0.47558045 0.7278483 -0.53558725 0.23050348 0.3884733 0.5668972 -0.42318904 -0.5938498 0.8083729 -0.72716963 0.57520676 -0.50956225 0.5343637 0.55660343; -0.7783528 0.85779625 0.7766646 -0.39537847 -0.87956244 0.87381476 0.80032986 -0.81597036 0.4796409 0.5052452 0.3675202 1.070989 -0.690381 -0.30790955 0.23048218 -0.8668406 0.5057694 -0.039287414 0.9482689 0.81575084 0.0988912 -0.24935482 0.38085127 0.6215717 -0.73503816 0.7708757 -0.5392362 0.7434906 -0.6566698 -0.7666758 0.75514185 -0.2231951 0.5467016 0.51421756 -0.22392362 0.89907426 -0.55615604 0.48091936 -0.2978689 -0.24816026 0.23326601 -0.6000167 0.75591457 -0.41842481 0.53686136 1.0035908 0.5028894 -1.0248363 0.905078 -0.597233 0.54220146 0.729651 -0.7724775 0.31902722 0.7395911 -0.6357299 -0.49710655 -0.562479 0.77258587 0.17295693 0.37122712 -0.5377766 0.11962055 0.40606344 -0.44178993 -0.81485194 -0.50133497 -0.65255 0.36446714 -0.8944873 -0.5259667 -0.2757228 0.690569 0.5453645 -0.40601206 0.6063261 0.52950406 0.73792166 -0.7183355 -0.5696452; -0.16335072 -0.099426076 0.5265241 -0.7768237 0.10783714 -0.4290263 -0.08906538 0.29246292 0.13187198 0.75816816 1.0297061 -0.35838348 -0.4760621 -0.77890503 -0.51912075 0.45946628 0.8572433 -0.89875054 0.02650084 -0.3100249 0.65508425 -0.76325566 1.0109452 0.55342406 0.3042365 0.22291999 -0.7853664 0.6135783 -0.74668735 -0.04095576 -0.29750744 -0.82916623 0.5493815 0.68688554 -0.81008023 -0.033660688 -0.8210836 0.73046184 -0.8780968 -0.64816695 0.89088434 -0.6458017 0.07239269 -0.821352 0.72755575 -0.10643859 0.9664241 0.18208116 0.23752426 -0.95470154 0.40573835 0.5771099 0.05555075 0.5966497 0.4851711 -0.6740047 -0.49890432 -0.7049067 -0.28225544 0.72479516 0.6629403 -0.47748208 -0.66927826 0.022298913 -0.6543012 -0.45082313 -0.25245368 -0.7759141 0.373255 -0.14335494 0.29023153 -0.8861285 -0.18650505 0.22310139 -0.67090106 0.82586205 -0.38261235 0.25006676 -0.6041666 -0.69201756; 0.30165416 -0.37319854 0.14158723 -0.6132097 0.36093113 -0.6080918 -0.41669136 0.6296644 -0.4570031 0.9087418 0.733201 -0.49392706 -0.07644342 -0.7271745 -0.77843153 0.30541965 0.7871986 -0.84760064 -0.25752756 -0.27813464 0.6861931 -0.69638646 0.550147 0.7257553 0.28410432 -0.5014527 -0.7087344 0.70028204 -0.69995385 0.23878059 -0.31713995 -0.6357927 1.0067724 0.87412286 -0.62985903 -0.26038957 -0.92677927 0.92082024 -0.62881374 -0.82025486 0.4913323 -0.7855262 -0.4408626 -0.63395673 0.86903465 -0.32637906 0.86718726 0.57513404 -0.31144717 -0.641741 -0.45860744 0.7737486 0.19062707 0.60525733 -0.16612843 -0.7486487 0.13047113 -0.79247797 -0.4470608 0.7620841 0.5860632 -0.24320105 -0.7773026 -0.3671825 -0.8479533 0.30224836 0.4283385 -0.34790388 -0.4402082 0.44397184 0.36613166 -0.9244026 -0.40263915 -0.379481 -0.65477705 0.67562866 -0.38704813 -0.4666101 -0.3384611 0.046537593; -0.36590093 -0.056809664 0.71502197 -0.73534507 -0.0648169 -0.4321501 0.31000566 0.28271046 0.25591296 0.5234196 0.67853814 -0.4916717 -0.6471024 -0.71496576 -0.7520661 0.33299994 0.9659328 -0.7054157 -0.050770737 -0.13335405 0.8815608 -0.73940295 0.87138534 0.56952757 0.18281487 0.19096963 -0.6678094 0.44627962 -0.7548804 0.012917675 -0.20902114 -0.7662226 0.53275293 0.9161994 -0.6903199 0.015411154 -0.6804812 0.66580945 -0.9572378 -0.7182707 0.5848537 -0.869308 -0.17362659 -0.7087042 0.6449765 0.10061632 0.8292758 0.1359347 0.12144698 -0.63915646 0.10990655 0.58000374 0.10911657 0.9018111 0.334832 -0.85479444 -0.43818903 -1.133445 -0.56673956 0.64392126 0.958269 -0.70725995 -0.7615096 -0.06579439 -0.77940387 -0.3555646 -0.12914163 -0.57977164 0.37286302 -0.18481323 0.30099812 -0.807292 -0.003102054 0.25417316 -0.66446334 0.99239296 -0.6463343 0.0070413235 -0.58460605 -0.7228561; -0.34712097 0.25688577 0.095746964 0.74654084 -0.71652836 0.477809 0.6670158 -0.26412296 0.023002317 -0.9608583 -0.9549351 0.65873057 0.32737815 0.78272015 0.48818624 -0.2696087 -0.6447182 0.8750356 0.69848746 0.12818563 -0.5005892 0.65087676 -0.8858273 -0.71536165 -0.039549146 0.24559067 0.68338263 -0.6874529 0.8314963 -0.27588752 0.6631529 0.29466146 -0.42512986 -1.0401301 0.69719434 0.63915306 0.39901686 -0.85680246 0.91913676 0.6782683 -0.9064129 0.5210979 0.34393087 0.35332775 -1.1007378 0.7652144 -1.0674118 -0.36536753 0.6164297 0.88704205 0.23778427 -0.046518262 -0.5019802 -0.52287287 -0.11073354 0.43079394 0.23818962 0.53660434 0.21563026 -0.7522625 -0.52218074 -0.06455336 0.96160483 0.019391002 0.4184895 -0.04891119 0.0673049 0.79669136 -0.04131459 -0.4278553 -0.24815013 1.116438 -0.026439136 0.354995 0.86839557 -0.68594515 0.19245332 0.33331525 -0.13659574 -0.04335341; 1.030964 -0.77338785 -0.60938394 -0.3199599 0.9414756 -0.74143624 -0.91034496 0.6074505 -0.54232985 -0.30516297 0.21154629 -0.5275524 0.6961487 -0.39980128 -0.47098938 0.5249121 -0.10492691 -0.3207378 -0.6838264 -0.7803789 0.32044253 -0.448465 0.43438867 0.000982338 0.5326028 -0.9711814 -0.17330371 -0.48190466 -0.09707207 0.7035897 -0.64195204 -0.57738703 -0.54431146 0.33509657 -0.24370125 -0.6917539 0.07690157 -0.5502152 -0.48056778 -0.35900092 0.50727504 -0.19777529 -0.64305484 -0.24946287 0.24073908 -0.62933946 0.2849696 0.5067657 -0.76282257 -0.08052275 -0.6986124 -1.0108805 0.77055836 0.4309184 -0.65078586 -0.18708324 0.72680277 -0.06916408 -0.2989908 0.48483488 0.30752322 0.94575644 -0.6322748 -0.5623185 -0.24844892 0.80749047 0.7526038 0.5173646 -0.7246457 0.55262154 0.8391701 -0.49537525 -0.675754 -0.60245115 -0.17894368 0.2951758 -0.52511525 -0.8654914 0.8768734 0.9585862; 0.6019006 -0.60537076 -0.66409355 -0.08655706 0.59295595 -0.7356671 -0.5452003 0.5787859 -1.0825679 -0.31961316 0.14678286 -0.5871489 0.7399228 -0.29467624 -0.33700484 0.7334125 0.25223178 -0.12042228 -0.5616173 -0.7493088 0.46151868 -0.18970722 0.28043282 -0.30987304 0.79066163 -0.73123646 -0.17843613 -0.46635437 0.10542296 0.83993375 -0.7728404 -0.41028714 -0.11486306 0.23389883 -0.27087513 -0.6629905 -0.27231857 -0.5715555 -0.18409505 -0.22026548 0.20221332 -0.5685064 -0.663232 -0.36541322 0.17337443 -0.7550816 0.05521666 0.65639585 -0.83467513 -0.07708701 -0.9724655 -0.6598551 0.70646137 0.3466706 -0.70535547 -0.09780296 0.9463569 -0.5069622 -0.5642399 0.6178395 0.29907456 0.7355935 -0.4607237 -0.9483248 -0.4359857 1.0626094 0.8287831 0.67612517 -0.9237049 0.78632617 0.67991126 0.14592406 -0.72758484 -0.755341 -0.13535605 0.2497937 -0.4262162 -0.75899184 0.8012509 0.6399505; 0.30267215 -0.3861378 0.341607 -0.8932591 0.3934429 -0.3222484 -0.13173 0.23778625 -0.19754687 0.85637665 0.5896629 -0.30519 -0.14765039 -0.9562067 -0.5183746 0.22518098 0.63661385 -0.8925694 -0.5322582 -0.4143222 0.64423233 -0.815544 0.7707687 0.9183804 0.29086074 0.015946068 -0.7733236 0.54169226 -0.92484933 0.33644307 -0.5796242 -0.54458284 0.6619667 0.9126392 -0.6201793 -0.38405174 -0.9562473 0.83481675 -0.7468587 -0.9349402 0.8841793 -0.5782625 -0.5291009 -0.640119 0.97347057 -0.265833 0.88266736 0.5705659 -0.47914657 -0.6521345 -0.10296136 0.7483685 0.42574656 0.5876349 0.04590124 -0.670125 0.026803056 -0.83246684 -0.4776708 0.6478246 0.8959475 -0.39602336 -0.5739977 -0.017063709 -0.57905525 0.14011464 0.0063306713 -0.6919984 -0.06265553 0.33888435 0.2689972 -0.87424207 -0.29734015 -0.21450487 -0.7958087 0.9361939 -0.44385326 -0.091751225 -0.43048683 -0.44573665; 0.5916995 -0.70464957 -0.7755386 0.17587921 0.6240368 -0.5590979 -0.5604042 0.93520904 -0.8592285 -0.65204275 -0.08455207 -0.48206574 0.52224517 -0.16054146 -0.3347269 1.0373818 -0.20230107 0.016216643 -0.9667695 -0.7823865 0.016317738 0.11156946 -0.35421762 -0.5381399 0.8769179 -0.6764597 0.19411154 -0.6183795 0.5864485 1.0666509 -0.6046831 -0.5650915 -0.2891851 -0.43646592 -0.13604803 -0.7846909 0.0104408 -0.6171554 -0.06262191 -0.059272945 0.17229311 -0.31348637 -0.7957886 -0.2287753 -0.39027786 -0.42674634 -0.42288888 0.7774889 -0.83855194 0.2565923 -0.7974151 -0.22585651 0.60026914 0.2994784 -0.6749067 0.26146185 0.9768411 -0.08276019 -0.87265235 0.3821485 0.094735846 0.5185477 -0.28586644 -0.9804604 0.016485972 1.0006342 0.944537 0.8485642 -0.8311003 0.9767073 1.0506049 0.7158243 -1.1422727 -0.76359576 0.199464 -0.3562544 -0.5199233 -0.6805886 0.593682 0.7423914; -0.18290219 0.23487315 -0.38698968 0.9370686 -0.20376514 0.30242267 0.3849416 -0.35627133 0.14664306 -0.9755795 -0.5636456 0.59518605 0.30161208 0.88692474 0.72180516 -0.3352703 -0.79973525 0.7410308 0.29699507 0.3181566 -0.50667465 0.8487277 -0.8858017 -0.9635871 -0.4430649 0.26241043 0.92072487 -0.7694756 0.7326618 -0.3303701 0.503409 0.67920136 -0.64920723 -0.8205949 0.644612 0.20205699 0.8785572 -0.61093473 0.8252194 0.7749175 -0.8318571 0.84393203 0.37899566 0.6611387 -0.9387861 0.42378405 -0.99695355 -0.3039155 0.50057375 0.92768735 0.13825719 -0.49345788 -0.4693738 -0.7770712 -0.25831932 0.6561806 -0.15987918 0.6776218 0.5654713 -0.52447283 -0.8618739 0.17504668 0.5322274 0.13166969 0.65990907 -0.28426272 -0.20786172 0.67979187 0.14833756 -0.34295782 -0.4465821 0.85221404 0.42545846 0.20294867 0.8886171 -0.892209 0.5958478 0.3764483 0.469039 0.3794259; 0.7635995 -0.85921264 -0.6084859 0.4940981 0.7075824 -0.7650322 -0.75454164 0.68493056 -0.73507875 -0.5245042 -0.391912 -0.4992237 0.58236516 0.19357578 -0.43192708 1.0265105 -0.6167501 0.15542485 -0.84372246 -0.7588121 -0.07669378 0.5999773 -0.5402197 -0.37741473 0.89863765 -0.8664136 0.34223658 -0.45051682 0.4132853 0.7245844 -0.65868217 -0.33998454 -0.6567245 -0.6026449 0.19200741 -0.7720922 0.59232384 -0.59984297 0.20511723 0.55570966 -0.31057417 0.2747522 -0.5652698 0.27852342 -0.62209266 -0.813327 -0.69337815 0.8605848 -0.6998423 0.5858783 -0.9323919 -0.6744087 0.5814683 0.14342482 -0.7190475 0.5491573 0.6689398 0.32147905 -0.9390384 -0.14398848 -0.3491998 0.6303374 0.07312081 -0.78101134 0.43900532 0.64356977 0.5855202 0.72172666 -0.6290638 0.6339062 0.8427611 0.5113576 -0.9004863 -0.6925275 0.6039224 -0.36173093 -0.32234588 -0.7584679 0.6887733 0.5736817; 0.5531125 -0.90833956 -0.633321 -0.24006093 0.59976286 -0.6247279 -0.582626 0.6425684 -1.0565565 -0.52907175 0.19997671 -0.6140625 0.83679837 -0.25710228 -0.6704748 0.95180523 0.25614643 -0.38034338 -0.84459776 -0.6482445 0.34356314 -0.2807941 0.034079004 -0.20040895 0.84389603 -0.923024 -0.342797 -0.5875467 -0.07826073 0.8280724 -0.7580639 -0.51287204 -0.08143991 0.1111705 -0.39828232 -0.68441296 -0.06938424 -0.43294045 -0.18307932 -0.42270508 0.25793958 -0.2930857 -0.80940914 -0.41133025 0.027765287 -0.48568952 0.079699986 0.6308153 -0.62986875 0.097427145 -1.0217931 -0.35060397 0.55701613 0.73482764 -1.012061 0.043275584 0.7838117 -0.50066894 -0.74701875 0.43684348 0.27489477 0.6010579 -0.37323314 -0.83378977 -0.5861218 1.0222197 0.94477016 0.8050814 -1.0414231 0.90389574 0.7679905 -0.0715364 -0.9135348 -0.66034824 0.051208686 0.018684402 -0.39912078 -0.94352335 0.44297048 0.80890894; -0.6648467 0.58693737 0.56812346 0.28469896 -0.8323506 0.40288585 0.72705007 -0.6025041 0.66213965 -0.58616364 -0.51612604 0.48144072 -0.84505254 0.49832103 0.47917035 -0.36602932 -0.516843 0.26652986 0.5912622 0.5415608 -0.35174164 0.39088777 -0.5847936 -0.6686746 -0.6586738 0.7688334 0.544361 -0.5393924 0.26246586 -0.5863934 0.6133638 0.5000736 -0.40826988 -0.5599998 0.58502823 0.45087856 0.6607825 -0.3616576 0.34016097 0.55058366 -0.4238151 0.3814713 0.6326835 0.3180326 -0.3532842 0.6591285 -0.4475968 -0.3933881 0.817936 0.44859737 0.49489477 -0.53161454 -0.7790984 -0.45320886 0.6247015 0.29832196 -0.91392964 0.69907635 0.47943923 -0.55108356 -0.53124785 -0.6248919 0.39225566 0.64321667 0.49389 -0.6197821 -0.7759228 -1.1958001 0.84255993 -0.75906485 -0.8057311 0.40430063 0.7522536 0.6049675 0.5999831 -0.31897107 0.49409065 0.7802622 -0.62493783 -0.67133814; -0.7087338 0.13682234 0.36190888 -0.77199066 -0.29915792 -0.24508655 0.6295207 0.4203596 0.15317754 0.7609129 0.5389632 -0.3654038 -0.55869687 -0.76205534 -0.85434675 0.45974308 0.71314675 -0.66232276 0.2144997 -0.14785263 0.89068264 -0.55512494 0.88089293 0.6990165 0.25300753 0.4216314 -1.0044807 0.76170176 -0.67105716 -0.3144712 -0.19232164 -0.80117315 0.6085463 0.5765879 -0.853644 0.3373408 -0.64305747 0.5093555 -0.6804221 -0.97494817 0.59640235 -0.85122615 0.0981653 -1.0285541 0.8995785 0.17588732 0.54979056 0.179429 0.5106376 -0.7539617 0.39942166 0.7377276 -0.44472086 1.044302 0.5105638 -0.91329205 -0.55428845 -0.86218685 -0.57599753 1.0376816 0.9266411 -0.54010344 -0.623338 0.17443804 -0.9262325 -0.46062148 -0.42504302 -0.506296 0.5316339 -0.13548298 -0.2214996 -0.48125848 0.21740754 0.1457518 -0.96044725 0.59054077 -0.4230229 0.6022862 -0.5858204 -0.64510113; -0.21178943 -0.14412804 0.40710542 -0.9419945 0.17018874 -0.5893167 0.15473224 0.21991709 0.11777375 0.82594824 0.8423563 -0.61636287 -0.3276676 -0.7963858 -0.797725 0.2768089 0.67332804 -0.94472784 -0.03653904 -0.40416718 0.7141836 -0.94086254 0.6730361 0.7467601 0.29334417 0.060519595 -0.8170989 0.7314331 -0.876122 0.009726279 -0.20667692 -0.66824377 0.5567519 0.8017883 -0.89368194 -0.050799478 -0.6273713 0.7360689 -0.5578735 -0.7661158 0.6637087 -0.9655133 -0.3056035 -0.71596533 0.9674932 -0.09337283 1.0296257 0.48744348 0.046798564 -0.7469278 0.05341871 0.27469692 0.10335645 0.55935067 0.6047173 -0.65089625 -0.3013538 -0.8120626 -0.5208095 0.9602427 0.95125186 -0.6903527 -0.7677143 -0.11267999 -0.75903815 -0.047464743 0.016479248 -0.67686355 0.487351 0.17346485 0.19527592 -0.7155787 -0.019663021 0.119717315 -0.84154063 0.88776225 -0.6158212 0.2391493 -0.61899555 -0.47061202; 0.12080787 -0.44348767 0.23117061 -0.9409497 0.3294125 -0.6249977 -0.24589205 0.5474728 -0.21864001 0.9224936 0.8211664 -0.5894534 -0.4507793 -0.8882084 -0.48719016 0.21397933 0.81689376 -0.7810914 -0.16742904 -0.5053584 0.72706085 -0.76160073 0.89597666 0.9342959 0.44508362 -0.013819161 -0.83608663 0.81017566 -0.87741536 0.20798908 -0.33584014 -0.5699923 0.64690846 0.62092423 -0.7850818 -0.2623039 -0.6861033 0.63864803 -0.77172595 -0.9889144 0.87610257 -0.8462547 -0.22990035 -0.923522 0.7081555 -0.4515503 0.8936667 0.24601018 -0.23497808 -0.95725363 0.014257987 0.39783412 0.022422355 0.66016084 0.31890747 -0.79338306 -0.031715762 -0.5572842 -0.42903826 0.8182179 0.75078636 -0.33351216 -0.7018182 -0.10822428 -0.6792068 0.23140162 0.0050405725 -0.72992456 0.12109493 0.27346542 0.42026246 -0.69947886 -0.18741252 -0.23913388 -1.0182527 0.8433455 -0.38514623 -0.06029913 -0.30195147 -0.33300954; -0.8059172 0.92447823 0.7010604 -0.17203452 -0.76575965 0.7502314 0.66134894 -0.63007534 0.9518231 0.55586874 0.123014055 0.665393 -0.52124447 0.13843004 0.48261297 -0.8132437 0.13277934 0.033053447 0.5468399 0.8628131 -0.11826804 0.14024122 0.1686046 0.024175012 -0.91660297 0.91852456 -0.0030428106 0.309441 -0.25273982 -0.92461294 0.48904493 0.48073307 0.4837945 0.28168476 0.3598708 0.8543303 0.14078212 0.67760026 0.30684927 0.29699782 -0.0660411 0.37772676 0.899721 0.5278647 0.13011447 0.47169513 0.32032517 -0.7570112 0.75884074 -0.054861918 1.0471075 0.2306638 -0.58936286 -0.4890144 0.7286322 -0.3535996 -1.1236669 0.10423337 0.9175323 -0.30741975 -0.41003242 -0.5376945 0.19409147 0.823824 0.35827604 -0.823681 -1.1683557 -0.63347936 1.1419388 -0.83417505 -0.8179336 -0.3043449 0.92571086 0.8733958 -0.24440934 0.22401568 0.64374816 0.80001074 -0.57662934 -0.38568977; -0.53016394 0.59121275 0.69252235 0.43059334 -0.534902 0.8883448 0.7178598 -0.39145926 0.63410616 -0.052716922 -0.3259263 0.68206304 -0.7505284 0.60519516 0.4273585 -0.804725 -0.38226178 0.5273743 0.78085524 0.7106594 -0.518633 0.468095 -0.51278794 -0.14711048 -0.4803187 0.8742196 0.17909479 0.6626393 0.45353967 -0.6830751 0.64593166 0.2489147 0.019000702 -0.5323597 0.26002955 0.75035703 0.48491418 0.47410622 0.46887362 0.3763395 -0.46964464 0.4020179 0.70717406 0.41533014 -0.47695935 0.8428542 -0.39458367 -0.5370187 0.7148921 0.34630248 0.7340027 0.8664492 -0.8642976 -0.5474497 0.8605124 0.30969286 -0.6000852 0.22590068 0.29165328 -0.3925836 -0.27585727 -0.9172611 0.34066638 0.54292905 0.34698302 -0.78011507 -0.61322093 -0.028948208 0.8332906 -0.5622186 -0.6497278 0.49492392 0.5169448 0.77486205 0.20546353 -0.3811106 0.62694275 0.58496344 -0.7798618 -0.8249669; 0.65619636 -0.6894287 -0.70372206 -0.065983 0.676506 -0.8808338 -0.8896672 0.60403943 -0.6696552 -0.31729078 0.0255679 -0.7905689 0.72620046 -0.47737616 -0.5339856 0.58825403 -0.3788131 -0.4918596 -0.6509455 -0.6821059 0.3962178 0.11833996 -0.03269948 -0.25593266 0.6495052 -0.8404903 0.26639426 -0.39439166 0.51407236 0.872403 -0.5852919 -0.34404027 -0.5362301 -0.14645375 -0.31227726 -1.034097 0.101462044 -0.4150724 -0.098292105 -0.19021443 0.22350392 -0.22639236 -0.7111111 -0.3683983 -0.051712114 -0.8153334 -0.13772339 0.7548588 -0.72409743 0.14387783 -0.87653047 -0.47332156 1.0116293 0.53470135 -0.67052335 0.40490705 0.88644636 -0.017119203 -0.6803082 0.39933816 0.19307163 0.8403013 -0.47353205 -0.97227037 -0.29524025 0.8300561 0.9204111 0.5319965 -0.77028036 0.8966802 0.92254364 0.26861653 -0.9226706 -0.7618148 0.16815215 -0.16975546 -0.3318168 -0.6487937 0.60490704 0.52577174; -0.7751391 0.66169363 0.71479154 -0.6315582 -0.6809003 0.8503859 0.63276494 -0.8893876 0.6517583 0.6012286 0.45448476 0.87892365 -0.59743506 -0.22827865 0.32029805 -0.9886821 0.6136972 -0.22854559 0.73746765 0.71151465 0.18031444 -0.41266778 0.4961218 0.48635003 -0.8283229 0.8254417 -0.4784671 0.5732909 -0.65304655 -0.7939162 0.81679696 0.13866772 0.6786653 0.58286965 -0.3771562 0.57270247 -0.5360719 0.44865912 -0.051516186 -0.21772644 0.20038457 -0.3381061 0.8539639 -0.095951 0.6170539 0.6081036 0.69147235 -0.6357626 0.716251 -0.6385817 0.70109606 0.5450045 -0.6711878 -0.16222222 0.6585692 -0.40000072 -0.7418796 -0.22120263 0.8809312 0.1433323 0.28698328 -0.45158345 -0.03448282 0.9240474 -0.4738788 -0.75125563 -0.62608093 -0.7873304 0.8175445 -0.7479149 -0.66367036 -0.6208549 0.71298105 0.8696593 -0.5825649 0.40442237 0.6218299 0.606342 -0.5715685 -0.50779927; -0.7400746 0.80833083 0.6368615 -0.46905935 -0.78446627 0.95859444 0.89562756 -0.5585828 0.7098652 0.65125155 0.40217605 0.740095 -0.7974041 0.12479544 0.56227565 -0.82954144 0.42238015 0.09427937 0.67571527 0.9442145 0.08715031 -0.14643146 0.41654852 0.268021 -0.9725149 0.6143674 -0.35470062 0.47077882 -0.5316606 -0.77510506 0.6160868 0.47631493 0.6576166 0.17588043 -0.019702595 0.6391721 -0.35231623 0.5391125 -0.058883023 -0.23042566 -0.05103454 -0.18976243 0.6822728 -0.040626258 0.41524717 0.5917906 0.49331945 -0.8373796 0.878746 -0.43717158 0.94834673 0.31466824 -0.6780933 -0.45995972 0.828181 -0.35516977 -0.7899343 -0.32845685 0.7906426 -0.29087707 0.12801841 -0.45526803 0.04861406 0.83408904 -0.10511011 -0.9628615 -0.9898377 -0.6449202 0.7037238 -0.8979248 -1.0135782 -0.39098465 0.73713684 0.7299575 -0.29519388 0.5496928 0.611116 0.5998028 -0.5218185 -0.44532868; -0.19358967 -0.15941162 0.42291963 -0.700127 0.20405072 -0.35755864 -0.2618009 0.24986978 -0.29680783 0.79591423 0.6580222 -0.4531668 -0.44990227 -0.56319517 -0.52770895 0.6020825 0.66331685 -0.61346483 -0.15979588 -0.44211775 0.5962425 -0.8740929 0.65198535 0.9912686 0.42985493 -0.15638177 -0.9753791 0.6378892 -0.7217815 0.02023627 -0.3316051 -0.84254736 0.9474795 0.8402222 -0.85453427 -0.14609197 -0.69891685 0.6813278 -0.8250432 -0.9381781 0.77480775 -0.8434479 -0.10903676 -0.5994443 0.68939424 -0.16505657 0.8712212 0.30763364 -0.28354517 -0.9545514 0.081945084 0.49511045 0.04531813 0.7122771 0.21180952 -0.7941162 -0.16818644 -0.80687684 -0.5143768 0.5375406 0.8559531 -0.40620497 -0.66508645 -0.00675543 -0.712381 0.06546242 0.015385094 -0.41696146 0.035655614 0.39025533 0.35181203 -0.88285035 -0.32361495 -0.20373571 -0.6938189 0.8601408 -0.4515854 -0.15967856 -0.47525245 -0.49495968; -0.1971823 0.3888159 -0.53242177 0.75245035 -0.28776547 0.3114837 0.27773234 -0.37719902 0.15133001 -0.7319241 -0.8995866 0.42255914 0.44663346 0.5734549 0.5281589 -0.3833614 -0.90593725 0.73898906 0.22959179 0.23417921 -0.74108636 0.7976405 -0.69846576 -0.80562234 -0.59748554 0.034184515 0.95646876 -0.5331277 0.935306 -0.35779998 0.25629783 0.7210579 -0.96208805 -0.591116 0.6142628 0.2675558 1.007791 -0.48164392 0.6221199 0.7984877 -0.73310095 0.98073655 0.25543392 0.6088024 -0.73956597 0.18908206 -0.89658386 -0.56206715 0.14933716 0.7013913 0.33056065 -0.84460896 -0.20025498 -0.55460244 -0.16705899 0.8840263 0.0956532 0.8546115 0.3507982 -0.73072577 -0.8061489 0.4019861 0.5863937 0.44002888 0.7153442 0.0060531665 -0.11278623 0.6356425 -0.1290332 -0.14517187 -0.20037739 0.6871827 0.20053415 0.12949589 0.8860668 -0.7729454 0.6417057 0.026758758 0.4030443 0.21437444; -0.50794655 0.10321189 0.5224344 -0.7432396 -0.35238022 -0.42170146 0.44569403 0.4027979 0.4134007 0.62146175 0.63896084 -0.5325934 -0.65711695 -1.0498922 -0.5569712 -0.024138838 0.63832015 -0.8748888 0.15231913 -0.17326982 0.78440815 -0.8861248 0.64741397 0.52003783 0.31645793 0.4207789 -0.6090366 0.50424105 -0.6650336 -0.15491742 -0.14490448 -0.89081436 0.60828507 0.7454338 -0.86644936 0.40274388 -0.7347309 0.53249156 -1.0233802 -0.8231183 0.9940789 -0.8411582 0.06494707 -0.92587817 0.63156384 0.2664003 0.6323404 0.30260628 0.28275532 -0.7792144 0.57773477 0.3622727 -0.49169743 0.63887686 0.52775025 -0.54675764 -0.38456434 -0.69543767 -0.5254578 0.88604766 0.98637074 -0.59534323 -0.6604926 0.4587188 -0.97196525 -0.2973483 -0.4266452 -0.44106436 0.54724324 -0.29275033 -0.0015977549 -0.5944168 0.3037606 0.13141236 -0.60120255 0.72883517 -0.76125205 0.4237632 -0.38049883 -0.68278927; 0.6218826 -0.83212614 -0.5086869 0.19171618 0.8551936 -0.5918643 -0.568541 0.7905533 -1.05727 -0.60368055 -0.06629578 -0.70558834 0.6081802 -0.37048364 -0.32526165 0.8532164 -0.2521162 -0.098670445 -0.73867476 -0.6849531 0.34800136 0.08534798 0.03474804 -0.25851336 0.96772707 -0.8789787 0.13793424 -0.4711199 0.5141093 1.0498524 -0.5580194 -0.48702613 -0.36220255 -0.12787053 -0.4417608 -0.84199876 0.3660327 -0.5453487 -0.20192896 -0.15410027 0.094916455 -0.26393706 -0.7538032 -0.18325336 0.024596466 -0.78663117 -0.23701183 0.62126064 -0.6415784 0.15704997 -0.76984614 -0.45334774 0.64918363 0.38249514 -0.7041773 0.4501805 0.90526444 -0.10041196 -0.58462787 0.4963481 0.3588022 0.66317147 -0.15565117 -1.0078995 -0.22187182 1.0248125 0.8064215 0.7690471 -0.91827327 0.9160186 0.8936477 0.391044 -0.8970397 -0.9042758 0.1707686 -0.3482118 -0.32888693 -0.90075636 0.6786819 0.63948673; 0.57604456 -0.8177164 -0.55419344 0.44524702 0.5886902 -0.7753788 -0.6729239 0.7056304 -0.64399046 -0.6086391 -0.5552341 -0.8058657 0.63129145 0.23867108 -0.35842448 0.96260786 -0.4496948 0.13214411 -0.71948075 -0.7980414 -0.40200153 0.5292948 -0.6412586 -0.54223347 0.7055953 -0.6588357 0.39199033 -0.47397044 0.5658999 0.7829984 -0.99124765 0.15129332 -0.52548313 -0.57089496 0.19436884 -0.6883398 0.40368596 -0.6678803 0.49201092 0.40219855 -0.3917191 0.30455592 -0.8243644 0.37782148 -0.41076085 -0.50464475 -0.5372966 0.6165259 -0.74116397 0.47910675 -0.5507306 -0.52340984 0.7019364 -0.10547721 -0.59342045 0.54708725 0.6400475 0.2834988 -0.8732295 -0.40894228 -0.409871 0.61630815 0.037465163 -0.82862645 0.5676493 0.61274564 0.54207873 0.5483844 -0.5977104 0.75294584 0.71909755 0.5460927 -0.8718135 -0.7686287 0.35082716 -0.34176213 -0.7050034 -0.53266 0.40960136 0.71472746; 0.9254428 -0.53095365 -0.93156475 -0.29450724 0.7272342 -0.7565288 -0.8714679 0.54820836 -0.46356314 0.47890586 0.6908432 -0.6000685 0.9822379 -0.52684575 -0.4908121 0.543623 0.32028553 -0.6532476 -0.7119424 -0.5618683 0.3712483 -0.51344347 0.6110984 0.21735887 0.4089272 -0.8217244 -0.26582643 -0.3074255 -0.22177723 0.50101036 -0.8893829 -0.28184676 0.3639827 0.55882317 -0.24218452 -0.85221153 -0.33080935 -0.3766631 -0.3585997 -0.22268482 0.3793751 -0.32862946 -0.73419076 -0.27881753 0.63408685 -0.8225395 0.6079581 0.710251 -0.6607453 -0.38940567 -0.5180214 -0.99658376 0.7849514 0.38006514 -0.8619632 -0.20406632 0.49455822 -0.30193704 -0.22679067 0.5182144 0.3774387 0.80121285 -0.5746615 -0.70902014 -0.30791494 0.7125717 0.5329322 0.08405505 -0.5789403 0.61596966 0.39265257 -0.5143214 -0.4365206 -0.80365807 -0.22719818 0.23947757 -0.50217265 -0.8396634 0.9264985 0.9178347; 0.565998 -0.9409974 -0.7916091 0.3156616 0.6581142 -1.0076743 -0.73356616 0.63624007 -0.82072693 -0.53021073 -0.18478304 -0.69926846 0.80711526 -0.168291 -0.45721316 0.7014214 -0.63824505 -0.14139737 -0.950652 -0.88776946 -0.065348655 0.44342968 -0.35361934 -0.4346258 0.72669196 -0.86063915 0.4466292 -0.7114083 0.44970885 0.89590764 -0.8926207 0.011435239 -0.7689262 -0.16532604 0.055467952 -0.58120507 0.34143177 -0.6039217 0.017706878 0.30891538 0.118615836 0.29272014 -0.8112651 0.37498677 -0.18715394 -0.7815926 -0.48533145 0.72415805 -0.60275817 0.6148435 -0.8242696 -0.6379551 0.93375605 0.2087073 -0.6236589 0.52612746 0.8144766 0.5502634 -0.60137355 0.08130516 -0.22387794 0.6437488 -0.16942288 -0.88551784 0.2151426 0.7577829 0.64909023 0.7975612 -0.43992603 0.61394036 0.9344765 0.35462514 -0.93534386 -0.78988105 0.51253366 -0.48264787 -0.36213127 -0.83310586 0.796849 0.56679994], bias = Float32[-0.015529476; -0.7256059; 0.7168833; 0.6703226; -0.78362644; 0.03901989; -0.43658847; 0.42249113; -0.5797554; -0.8413509; 0.7846166; -0.7712585; -0.36294755; 0.21753253; 0.5137484; 0.14588177; 0.015424962; -0.37022847; 0.5253725; -0.5061441; -0.03545559; 0.6279705; 0.7975671; -0.07034765; -0.7673932; 0.8703869; 0.43890554; 0.41790956; 0.85832083; 0.8036127; 0.12697889; -0.81352025; -0.80967337; -0.35362685; 0.30065775; 0.3216815; -0.42059505; 0.8372364; -0.35243806; 0.5543911; -0.43236798; 0.7637857; -0.79004836; -0.055746518; -0.64780426; 0.30059028; -0.78892076; 0.4899541; 0.76225364; -0.8091825; -0.47567505; 0.70398945; -0.37966156; -0.8515585; -0.70479923; -0.8457195; 0.97153926; -0.3629585; -0.14720069; -0.801746; 0.1912806; 0.74687934; 0.46532282; -0.109616056; 0.44107312; -0.6506963; -0.8161442; -0.7939894; -0.16530022; 0.45920107; 0.016204253; -0.48918602; -0.3563699; -0.81506795; 0.7727356; -0.76895285; 0.14332858; 0.5655532; -0.53221864; 0.38117263]), layer_3 = (weight = Float32[-0.36457172 0.32865506 0.42765424 -0.78771806 0.63584244 -0.2603471 0.14444758 -0.35100487 -0.50644755 0.00091739465 -0.707 0.123417504 0.7023612 0.55260587 -0.808067 -0.23636925 -0.3020036 -0.24952747 0.39496347 0.31662 -0.77066416 0.42969447 0.1042431 -0.30155578 1.0264097 0.5642662 0.033212103 -0.6217573 0.62019426 -0.080962986 -0.66309786 0.40914822 0.3894719 -0.0069380375 -0.054287013 0.48526374 -0.10005282 -0.3121228 0.35228106 -0.48776776 0.19216935 -0.40806514 0.396859 0.5856543 -0.56556916 0.19708242 0.7373318 -0.8860318 -1.0861267 0.16851766 0.8805488 0.33572644 -0.26570243 0.08266941 1.0561284 -0.62531304 -0.9457761 -0.20937069 0.044389073 0.66675574 -0.6839062 -1.0285156 -0.8851651 -0.64857024 0.2236508 0.7068124 0.34066415 1.055493 0.56281954 0.5870524 0.5000695 0.90589 0.43875074 -0.625369 0.07399367 0.06228555 -0.14581169 0.5597075 -0.5798255 -0.09079072; -0.5336554 -0.34984714 -0.3178965 0.15193488 -0.7431177 0.44247657 0.4111847 -0.5652835 -0.2542314 0.7606675 0.8020022 -0.8967685 0.70413595 0.29196802 0.3034787 0.5203308 -0.41783252 -0.8677605 0.4006184 -0.55322295 0.025880788 -0.10065168 0.8395966 0.74688554 -0.43419805 -0.21478166 1.3714609 -0.5993328 -0.13416529 0.70060664 0.9618887 -0.7323608 -0.8315892 -1.1040561 0.6622035 0.4649551 0.51154065 0.63709915 0.5040368 0.46360725 -1.3400041 -0.90947 -0.7049043 -0.19134817 0.14979683 -0.74813735 -0.39708427 0.21935004 0.014403835 -0.7935353 0.047432814 0.47396827 0.84487116 0.6544158 -0.20834602 0.25666162 0.5824046 0.6098791 0.53852767 -0.79868615 0.40097347 0.039610982 0.11662093 -0.052758295 -0.60800916 0.6574299 -0.7315227 -0.10891027 -0.076255165 -0.10557638 0.5051376 -0.1347601 -0.45114315 -0.0025640714 -0.7111825 -0.6621846 0.47838256 0.25199118 -0.008250485 -1.0045863], bias = Float32[-0.7922723; 0.21831562]))
loss_neuralode(p_trained)


#Moderate noise collection of loss
#----------
#-----------------------------------#
rng = Random.default_rng()
Random.seed!(99)

x1=Array(sol)

x1_mean = mean(x1, dims = 2)
noise_magnitude = 7e-2
x1_noise = x1 .+ (noise_magnitude*x1) .* randn(eltype(x1), size(x1))
#Displaying true data vs noisy data
dudt2 = Lux.Chain(Lux.Dense(2, 80, tanh),Lux.Dense(80, 80, tanh), Lux.Dense(80, 2))
#Setting up the NN parameters randomly using the rng instance
p, st = Lux.setup(rng, dudt2)


true_data= x1_noise[:,1:end-10]
### Define loss function as the difference between actual ground truth data and Neural ODE prediction
function loss_neuralode(p)
    pred = predict_neuralode(p)
    loss = sum(abs2, true_data .- pred)
    return loss, pred
end

p_trained_moderatenoise = (layer_1 = (weight = Float32[-0.58593476 -1.524755; -0.65538114 -0.7590612; -0.34794694 -0.28773555; 0.40748823 0.44292468; 0.4275936 0.4516364; 0.8006977 0.5587954; 0.4052721 0.7664945; -0.47447562 -1.079935; -0.62335646 -0.571499; -0.90737253 -1.0583795; 0.7871532 0.62637097; 0.6760149 1.0042299; -0.5950861 -0.69386125; -0.3749787 -0.64250994; 0.77088916 0.63384664; -0.66801465 -0.4696443; 0.1608726 0.34804788; -0.41949236 -0.34682953; -0.58093816 -0.92546654; -0.419818 -0.63009244; 0.102960005 0.7749718; -0.57717615 -0.7953494; -0.895251 0.556662; -0.7397578 -0.37727687; -0.6592886 -0.8328025; -0.1748795 0.2887965; -0.6605009 -0.9432878; -0.6901912 -0.19511321; -0.83613247 -0.8327695; 0.013485824 0.2968258; -0.626247 -0.44575426; -0.71884197 -0.35355586; -0.695366 0.13074768; -0.34527394 -0.3059143; 0.42144004 0.48958355; -0.7030251 -0.39717293; -0.69237596 -0.5884913; -0.74833834 -0.32788694; 0.42216817 0.6168567; -0.5146451 -0.29010043; -0.364942 -0.31094965; -0.34560466 -0.46266013; -0.655056 -0.86297965; -0.5461455 -0.5449384; 0.36874762 0.5702408; -0.57506365 -0.46527353; -0.47899073 -0.7927859; 0.6234676 0.6009772; -0.7288108 -0.37091312; 0.5332969 1.3152761; 0.3951159 0.8826494; 0.4683482 0.6029835; -0.51834995 -0.48528582; -0.41623935 -0.63355976; -0.5428726 -0.78250545; -0.80573374 -0.8543428; -0.62828386 -0.54878086; 0.54296494 0.8830623; -0.08894288 0.29771966; 0.44252193 0.5692706; -0.40962154 -0.6371338; -0.37717116 -0.31218183; -0.5678172 -0.3021095; -0.75567657 -0.5937625; 0.8880679 -0.58315176; 0.6607061 0.5001076; -0.703454 -0.6607596; -0.6589828 -0.6236293; -0.6045993 -0.51913923; 0.6157424 0.8417803; -0.3638413 -0.37755415; 0.69534105 0.79195344; 0.57366484 0.20379785; 0.59727144 0.64433235; -0.6844475 -0.51901406; -0.58902955 -0.5488377; 0.8740755 0.79118997; 0.38318643 0.38394025; 0.5442415 0.62989134; 0.69103914 0.8378629], bias = Float32[-0.5788515; -0.5811164; -0.6223141; 0.60005826; 0.5829786; 0.63230413; 0.6552959; -0.4995893; -0.5437151; -0.31907088; 0.63262016; 0.40141854; -0.6506847; -0.53813326; 0.5618125; -0.5957836; 0.7848359; -0.593558; -0.60286975; -0.574336; -0.41178438; -0.62204593; -0.5623279; -0.61857104; -0.45405746; 0.13924472; -0.6657859; -0.5211408; -0.58905274; -0.4609525; -0.55136067; -0.29434583; -0.696795; -0.5764255; 0.54324996; -0.59803975; -0.65274745; -0.57682425; 0.5812871; -0.51160353; -0.57044977; -0.58192134; -0.6146577; -0.5768683; 0.5697648; -0.60358423; -0.57082796; 0.5817214; -0.6288796; 0.55040306; 0.5929715; 0.56055653; -0.56851166; -0.55162454; -0.6585659; -0.62317145; -0.54551595; 0.61480576; -0.7075837; 0.58359075; -0.55605376; -0.65147346; -0.5253242; -0.55831814; 0.38961875; 0.61196554; -0.6291237; -0.62076724; -0.5334851; 0.65096337; -0.5845875; 0.532683; 0.5092368; 0.7195607; -0.62243766; -0.55561876; 0.62318987; 0.5727571; 0.5526196; 0.6194071]), layer_2 = (weight = Float32[0.5983871 0.6370443 -0.6009691 0.46677187 0.5644898 -0.3891553 -0.7566927 0.50382495 -0.538883 0.65678626 -0.61057657 -0.57617724 0.48638088 -0.5445316 -0.50889194 -0.48453712 0.49586615 -0.48129243 0.54103404 -0.60172164 -0.4165028 0.49202716 0.491495 -0.4792819 0.39644513 0.5591316 0.47790906 -0.6974504 0.5780637 0.5489671 -0.47231868 -0.49574217 0.6288991 -0.57774293 0.5204207 -0.5584991 0.49951318 -0.7338701 0.53001684 -0.7434721 -0.66089576 -0.5673244 0.438507 -0.4285921 0.46393922 -0.75642467 0.54157966 0.578297 -0.5492233 -0.7229896 -0.7826206 0.7113123 -0.5637785 -0.6406008 0.703357 0.60365605 -0.48173013 -0.6833898 0.5732157 0.41715497 -0.7170438 -0.69015056 -0.7880137 0.45366332 -0.5065218 0.5562255 0.62337506 0.47546294 -0.5363362 -0.4156269 -0.6372124 -0.76014596 0.7857705 -0.71160525 -0.30643353 -0.71761715 -0.706629 0.542589 0.62025005 -0.49868408; 0.7560407 0.5071204 -0.6551121 0.5277253 0.42066684 -0.45906672 0.14121728 0.6194654 0.4629392 0.6847327 -0.7047674 -0.6606441 0.683243 -0.5134606 -0.5347062 -0.2277543 0.7586919 -0.5735818 0.59510326 -0.69576436 -0.47646335 0.7229446 0.5854973 -0.6008573 0.4581431 0.5247278 0.6225234 -0.65634066 0.75337577 0.55606383 -0.5948897 -0.46159366 0.62358755 -0.586835 0.5719132 -0.6780368 0.7410505 -0.6496648 0.49764302 -0.69566405 -0.50028044 -0.41245154 0.44179094 -0.44163436 0.54866415 -0.7659193 0.6860034 0.6931582 -0.5892037 -0.7215563 -0.8228123 0.5272633 -0.4136189 -0.5161345 0.56484586 0.68268746 -0.6755294 -0.77660066 0.42883512 0.633304 -0.67578244 -0.67030907 -0.7446395 0.7040163 -0.6644244 -0.72273546 0.76778346 0.719942 -0.39898127 -0.8050367 -0.67293745 -0.6055219 0.6325786 0.43491566 0.69090843 -0.60989636 -0.52795357 0.6317522 0.4292289 -0.7342646; -0.57595474 -0.5023217 0.68300796 -0.6420662 -0.5947153 -0.6953608 0.7536277 -0.5818207 0.5394135 -0.45512226 0.5669628 0.6073298 -0.63427114 -0.6952234 0.14263602 0.60465926 -0.72889185 0.46311915 -0.5979113 -0.7338597 0.49699062 -0.6287117 0.7748045 0.7009568 -0.6632854 -0.52049917 -0.71990097 0.6900995 -0.74436986 -0.51580197 0.46938878 0.46168268 -0.36805466 0.4827531 -0.80976576 0.4560429 0.66971856 0.46923688 0.5525986 0.62470037 0.7904398 0.71638346 -0.506136 0.56290555 -0.60046077 0.49981436 -0.7096339 0.6608391 0.47613734 0.43737632 0.5287621 0.43041196 0.737334 -0.72341084 -0.75127983 -0.6260349 0.7583419 0.631455 -0.49604315 -0.69192064 -0.44016004 0.74789804 0.71772665 0.52478766 -1.0040885 -0.64235264 -0.58937156 -0.15424716 0.5441307 0.48560643 0.72923297 0.38872504 -0.46478644 0.60061055 0.506724 0.7973344 0.6938939 -0.71335167 0.6164233 0.7544241; 0.49867982 0.38146034 -0.42073193 0.7569465 0.5982126 -0.17428552 -0.70446646 0.36792243 -0.9118556 0.49775746 -0.55466413 -0.4927912 0.54116666 -0.7102115 -0.43430212 -0.66110253 0.7237091 -0.6026964 0.52020186 -0.6245844 -0.65826833 0.6004627 -0.62747526 -0.73791176 0.37571076 0.5209559 0.65955776 -0.45847577 0.5703237 0.7494959 -0.5503584 -0.7629608 0.6955017 -0.80062574 0.48259312 -0.7460338 0.36408135 -0.603593 0.6608068 -0.7180786 -0.6908337 -0.5422889 0.38477212 -0.5250272 0.6258529 -0.4957297 0.66289616 -0.1629586 -0.55040395 -0.6133766 -0.41339812 0.647463 -0.64830256 -0.88304746 0.58058 0.62866074 -0.600531 -0.5311883 0.46197996 0.70670384 0.05093539 -0.44875443 -0.4945922 -0.5167023 0.115342535 0.7549086 0.39177588 0.10687406 -0.6255899 -0.7069518 -0.5676498 -0.4156055 0.51226723 -0.5127015 -0.7174711 -0.81101155 -0.33993274 0.51809675 -0.19784643 -0.5874462; 0.8032605 0.8189614 -0.66313565 0.4477579 0.48331982 -0.62813294 0.478954 0.7323585 0.65846556 0.62671226 -0.58402115 -0.7926719 0.85732687 -0.5780209 -0.6965611 0.57887715 0.5553489 -0.47747344 0.48149166 -0.6253152 -0.41049737 0.74631256 0.6876496 0.94196486 0.6589995 0.72328687 0.7186805 -0.52293694 0.7768635 0.56231505 -0.5868015 0.75718546 0.74709576 -0.5419733 0.44842067 1.0152882 0.59717035 0.29165915 0.39788848 -0.5281561 -0.58707553 -0.51876116 0.46149278 -0.60198665 0.6345264 -0.6873246 0.931407 0.6309428 0.8199236 -0.6562579 0.7158703 0.3993306 -0.44598103 -0.51509196 0.6156534 0.7966812 -0.26939353 -0.45244372 0.0024827118 0.5137825 -0.7258112 -0.43719962 -0.58883095 0.7352423 -0.8451686 -0.670587 0.7995981 0.47791874 -0.36360738 -0.70619345 -0.5386062 -0.5991077 0.64950883 0.71595496 0.705882 -0.47738716 -0.6866672 0.56618595 0.5738948 -0.44026378; -0.3910275 -0.65022767 0.43585214 -0.4932283 -0.724806 0.70517576 0.7511819 -0.5439851 0.66022325 -0.5123023 0.6585259 0.7036515 -0.60066086 0.54108095 0.6375446 0.7008055 -0.54408664 0.6031927 -0.3923521 0.4268697 0.7555127 -0.62463 -0.71694374 0.68649834 -0.57802033 -0.77879584 -0.73393285 0.42013377 -0.40526772 -0.43666667 0.52414566 0.6625899 -0.5413318 0.7244435 -0.78143287 0.75272965 -0.7268237 0.44461092 -0.66316706 0.7480006 0.5953444 0.6571781 -0.67934257 0.52319247 -0.5361342 0.4478604 -0.5027048 -0.7141869 0.7036066 0.73477757 0.45016667 -0.7739148 0.5364916 0.6554327 -0.55484813 -0.4145497 0.453623 0.62116027 -0.775113 -0.47561312 0.45592988 0.5099733 0.58105874 -0.4221879 0.47797772 -0.5479373 -0.4231703 -0.34617123 0.6055037 0.46424598 0.45065036 0.41673672 -0.49208614 0.4019618 -0.32134128 0.7106199 0.7384594 -0.67508155 -0.7341978 0.6039014; 0.6158945 0.48543918 -0.56473166 0.4335744 0.42758188 -0.7799479 -0.57993644 0.64434123 1.4369091 0.7225233 -0.5030656 -0.73880386 0.49679416 -0.53814286 -0.63420665 -0.7184684 0.42665237 -0.62712073 0.66626436 -0.54721355 -0.51846665 0.80069804 0.6646067 -0.51455826 0.6838075 0.7899917 0.45705798 -0.48309958 0.68776226 0.51515275 -0.68991774 -0.6053701 0.44925606 -0.66176087 0.4456659 -0.61480963 0.79496646 -0.5540055 0.5114075 -0.39835408 -0.3986935 -0.60436994 0.6861994 -0.44776106 0.57767034 -0.6597642 0.5446943 0.37461475 -0.46587184 -0.6355478 -0.6380356 0.68223006 -0.48725146 -0.7145356 0.53662276 0.46354088 -0.532462 -0.416153 0.5969133 0.62670505 -0.6906215 -0.40814596 -0.66736466 0.591042 -0.97109133 -0.46944794 0.5005677 0.7754676 -0.44267175 -0.79018307 -0.767876 -0.4841069 0.60827637 -0.87195617 0.8538462 -0.41078848 -0.75387216 0.61759615 0.60013765 -0.7409646; 0.5903579 0.4609459 -0.6060064 0.5910047 0.5579301 -0.661853 -0.7881667 0.65165323 -0.43030006 0.47264317 -0.78530353 -0.7256813 0.56520915 -0.47527033 -0.6879503 -0.37421075 0.40905148 -0.40536374 0.50580347 -0.37307394 -0.40256438 0.56912994 0.85614645 -0.38750032 0.48260918 0.5764528 0.6826506 -0.61481094 0.43968332 0.5429033 -0.7086556 -0.43987152 0.8384577 -0.7447878 0.5736142 -0.4214417 0.85354763 -0.7523912 0.57667655 -0.61088145 -0.40399733 -0.70690316 0.6370642 -0.36057967 0.5073403 -0.37474316 0.50113666 -0.14315288 -0.41128123 -0.7628337 -0.75366 0.40912136 -0.3905773 -0.35659072 0.73907065 0.56964433 -0.25650156 -0.6989411 0.5874529 0.5684705 -0.70792776 -0.65245634 -0.6921244 0.6506766 -0.8913654 0.30025595 0.47816372 0.653287 -0.50432986 -0.5154994 -0.64448494 -0.4233052 0.42474946 -0.61311257 -0.16182141 -0.61259145 -0.55705833 0.45447943 0.3465872 -0.5887174; -0.6776579 -0.59869206 0.57144773 -0.50566834 -0.705406 0.7447669 0.5034099 -0.61303186 -0.65749604 -0.5908892 0.6501818 0.56398225 -0.6502248 0.4358435 0.45104754 0.69490564 -0.55009127 0.5802827 -0.47891125 0.4911366 0.46192604 -0.58295137 -0.5650437 0.6068022 -0.6869028 -0.5726659 -0.4254156 0.7777999 -0.4658305 -0.49725944 0.42544836 0.45258513 -0.78858864 0.45889163 -0.43460494 0.49808592 -0.4833295 0.7008262 -0.44038603 0.6919075 0.54350394 0.7742769 -0.58408135 0.6314779 -0.7154835 0.47407696 -0.41819474 -0.54432184 0.651093 0.5625579 0.68011844 -0.502162 0.68035114 0.5059261 -0.63719684 -0.7425933 0.45790917 0.5769412 -0.5697762 -0.69973844 0.49440354 0.59234136 0.5235258 -0.50996065 0.6596174 0.68391186 -0.45089337 -0.41041523 0.64220524 0.51387405 0.62255263 0.7715895 -0.66505426 -0.7378607 -0.71799487 0.6825785 0.6321572 -0.7789374 -0.6542103 0.6166164; -0.4973527 -0.54383034 0.74522847 -0.63488245 -0.5882777 0.71864694 -0.776953 -0.7194387 -0.36863405 -0.5472213 0.6485102 0.49868917 -0.5070728 0.5462256 0.3933111 0.4745055 -0.7541776 0.5653648 -0.44506854 0.56698406 0.40987176 -0.6817301 -0.63942444 0.5132883 -0.5226334 -0.76162606 -0.40058982 0.6160134 -0.6634926 -0.80594707 0.7594401 0.45575747 -0.7496701 0.4722057 -0.4225944 0.6022542 -0.5419009 0.5988484 -0.5652895 0.50559753 0.74201185 0.59125364 -0.49501798 0.5419699 -0.6429667 0.6882489 -0.67011005 -0.59756297 0.6271758 0.480987 0.3998758 -0.42823234 0.5238171 0.6224767 -0.4120678 -0.6673301 0.6214972 0.52690893 -0.41528735 -0.77739966 0.7487891 0.48805726 0.43114322 -0.5519102 0.6216638 0.57772785 -0.5918147 -0.5311561 0.6792297 0.737811 0.48583934 0.4039101 -0.5244368 -0.73211044 -0.563753 0.5671042 0.7595513 -0.78403354 -0.7194247 0.48397964; 0.45927596 0.6341926 -0.4896519 0.6557956 0.40921575 -0.64726824 -0.75749266 0.5156261 0.9094264 0.44242212 -0.6563182 -0.43056065 0.7676309 -0.6142392 -0.6522923 -0.21993797 0.71288735 -0.46208468 0.47569394 -0.55723655 -0.5759828 0.7570247 0.61746037 -0.39150733 0.6500356 0.47150937 0.44565955 -0.6815388 0.58402294 0.4816971 -0.40819898 -0.483937 0.8145835 -0.49748272 0.51818156 -0.42996246 0.6043463 -0.52741647 0.4996026 -0.47821665 -0.63551754 -0.6982066 0.7569294 -0.5451399 0.48493484 -0.41310567 0.46224198 0.40091762 -0.48576972 -0.53486496 -0.7001073 0.5906707 -0.5397513 -0.7407371 0.7768305 0.6982446 -0.4738136 -0.6125947 0.7255702 0.5205586 -0.6905746 -0.49010292 -0.48407018 0.8271113 -0.7895611 -0.90168184 0.62659544 0.7048471 -0.5604909 -0.76361334 -0.5845183 -0.42464218 0.6507073 0.077315606 0.55403894 -0.53622526 -0.63658595 0.5877773 0.555765 -0.42625344; 0.7230384 0.4793269 -0.7287753 0.55974054 0.5214865 -0.755305 -0.71778893 0.55713785 0.757842 0.4446508 -0.6747185 -0.5572019 0.56421614 -0.47444895 -0.7932599 -0.59005266 0.7086391 -0.49633867 0.6239077 -0.5582103 -0.5982284 0.5392138 0.6934783 -0.6275405 0.57675093 0.42558676 0.70994556 -0.77323824 0.5796747 0.6435135 -0.44365743 -0.5350949 0.56431824 -0.74815965 0.6276171 -0.49803218 0.5860214 -0.5385192 0.42293614 -0.66582465 -0.49453714 -0.69399345 0.5542315 -0.46839535 0.54104537 -0.43403566 0.59624726 0.7034158 -0.504608 -0.6344089 -0.77760065 0.4427519 -0.43037668 -0.6853842 0.77977943 0.5700498 -0.68994576 -0.6084947 0.6852271 0.41117495 -0.6235379 -0.5982317 -0.7815809 0.574327 -0.5662902 -0.94796294 0.5109562 0.73281 -0.405672 -0.4949213 -0.42828503 -0.65898395 0.54107374 -0.86967164 0.77753776 -0.6894729 -0.7625371 0.65151095 0.6392087 -0.5754716; -0.50669545 -0.40644494 0.61695623 -0.6058045 -0.47633585 0.56950665 -0.8211464 -0.6174748 -0.36861095 -0.7315583 0.47721356 0.52378416 -0.763774 0.6107051 0.5148209 0.8141331 -0.69529396 0.52117133 -0.55775636 0.6219846 0.66670656 -0.55300254 -0.45028383 0.76026803 -0.5942519 -0.43523425 -0.58903986 0.7788032 -0.55159396 -0.76997375 0.72254354 0.5617145 -0.44101626 0.6780371 -0.67911506 0.55317867 -0.65415007 0.6569108 -0.64459145 0.5051707 0.44324663 0.7592282 -0.44731638 0.65557253 -0.42493328 0.7322896 -0.47656357 -0.7540199 0.43757665 0.53128207 0.64584714 -0.69642687 0.4465664 0.45624042 -0.47622347 -0.75679713 0.7636308 0.6847897 -0.58415955 -0.63890094 0.7108367 0.55312526 0.50338244 -0.50203794 0.5933418 0.6569551 -0.5233538 -0.7720773 0.563351 0.78699946 0.5983944 0.5997669 -0.42498115 -0.49448532 -0.64328974 0.5037608 0.70505977 -0.6132577 -0.47640854 0.5516657; -0.816372 -0.72360253 0.47580132 -0.39931753 -0.7128964 0.79964554 0.6196456 -0.5624944 -0.6733303 -0.49398613 0.6567149 0.7501448 -0.8286876 0.5866423 0.7278045 0.3059334 -0.6852947 0.7441239 -0.75205475 0.4741253 0.70981747 -0.5170948 -0.7422381 0.632961 -0.51217675 -0.73041564 -0.52817655 0.6952369 -0.72494555 -0.63622236 0.62295437 0.69046116 -0.46421057 0.74340546 -0.5739229 0.7556428 -0.58025676 0.49321327 -0.7610649 0.6049701 0.69664824 0.6774334 -0.6572152 0.49207133 -0.5234494 0.4058116 -0.5005782 -0.5038519 0.6559008 0.50160503 0.8222016 -0.63102317 0.421957 0.61904734 -0.48499635 -0.6670084 0.53963745 0.6983472 -0.6872031 -0.7541099 0.5927286 0.62118983 0.691393 -0.77417487 0.46130925 0.64102787 -0.6449998 -0.80013484 0.6527108 0.7321202 0.7159269 0.4993395 -0.53336453 -0.3681427 -0.7771243 0.58339316 0.735417 -0.5646495 -0.5448898 0.45087922; 0.4949677 0.49637505 -0.6957556 0.5915622 0.49734786 -0.53459626 0.71907306 0.63085777 0.5946187 0.42011842 -0.64921236 -0.4567073 0.5510948 -0.56005496 -0.4337381 -0.51006925 0.45110145 -0.49481687 0.6857002 -0.5645333 -0.5700204 0.75940585 0.73043555 -0.44266346 0.6700773 0.77305263 0.54932857 -0.6197671 0.5225543 0.738446 -0.4563341 -0.6125968 0.61936957 -0.574186 0.6044746 -0.5710739 0.6653002 -0.5409069 0.55482197 -0.7934983 -0.56292045 -0.7896332 0.6522213 -0.738442 0.50253356 -0.518423 0.52511054 0.41402578 -0.48159817 -0.53970647 -0.69115657 0.47512808 -0.759447 -0.65938294 0.688564 0.59461254 -0.48487487 -0.49383238 0.539002 0.47116977 -0.63642263 -0.5077066 -0.45886803 0.6101965 -0.6339154 -0.43596166 0.4866111 0.7047843 -0.50065607 -0.53319675 -0.51620775 -0.40472457 0.49702188 0.5824425 0.69595414 -0.5994601 -0.73159933 0.43106788 0.69635487 -0.59060603; -0.63482755 -0.3719207 0.51626986 -0.47267216 -0.716226 0.5344787 0.77044725 -0.4933845 0.6980411 -0.7510598 0.4658475 0.7242717 -0.698956 0.45146817 0.6967734 0.6037871 -0.77431303 0.41130927 -0.4081609 0.61272043 0.59837824 -0.7548348 -0.11841845 0.43233928 -0.6897403 -0.44812566 -0.566157 0.45013475 -0.607773 -0.54786354 0.7736047 0.5563996 -0.5693215 0.7441349 -0.7057391 0.7405676 -0.54903656 0.6200402 -0.41905922 0.6083803 0.47390965 0.62213767 -0.5659602 0.7284706 -0.6454161 0.7254439 -0.47414693 0.512982 0.786817 0.5590591 0.7536839 -0.6816726 0.7757707 0.7974829 -0.7120042 -0.653716 0.84155846 0.58768636 -0.44867292 -0.631334 0.5419567 0.6208478 0.45410857 -0.1287674 0.14906248 -0.65853715 -0.5230055 -0.36205304 0.64263207 0.5200157 0.76161337 0.6111897 -0.708002 0.43201175 0.75737697 0.547366 0.41710788 -0.6501708 0.15428148 0.40728313; -0.46398363 -0.7337632 0.50518614 -0.54090846 -0.4210803 0.52950126 0.74854934 -0.8099613 -0.54080045 -0.42781246 0.60957724 0.7422355 -0.8289659 0.57948095 0.5966954 0.3235685 -0.43644714 0.44127128 -0.42343548 0.5979482 0.4015271 -0.47336012 -0.67479014 0.39474425 -0.4791411 -0.5139635 -0.50882727 0.49757528 -0.4389492 -0.66420895 0.4743317 0.6659152 -0.7633329 0.3939212 -0.42792973 0.39618608 -0.46708605 0.6749448 -0.5582876 0.63643205 0.6176274 0.7548948 -0.702996 0.7593102 -0.42526147 0.5729449 -0.53267586 -0.44474298 0.3684141 0.56281525 0.65553594 -0.67038715 0.432064 0.4972913 -0.66308975 -0.54044384 0.63865006 0.48761576 -0.7182375 -0.3961394 0.7107985 0.4403136 0.77128386 -0.75507975 0.6412083 0.705159 -0.46663973 -0.7157977 0.45634854 0.6654406 0.53881377 0.60510087 -0.66372234 -0.23494293 -0.7954798 0.68694717 0.60123867 -0.6114263 -0.42306477 0.47422642; -0.68104833 -0.62272304 0.5407877 -0.72050774 -0.46309036 0.54075205 -0.69535446 -0.76651317 -0.42187607 -0.5832488 0.631111 0.51973486 -0.6095166 0.52713406 0.45670313 -0.58704317 -0.48645514 0.4696294 -0.6217259 0.5455584 0.43239185 -0.5177634 -0.5874299 0.13592625 -0.5730772 -0.64300036 -0.505739 0.44320193 -0.42688173 -0.6017518 0.41984165 0.7630815 -0.80887854 0.59519076 -0.7478568 0.60873365 -0.4873011 0.52210015 -0.5422129 0.5487334 0.50282776 0.61026853 -0.7259565 0.7464393 -0.48355278 0.5516585 -0.58792114 -0.68885577 -0.4445685 0.40728444 0.45122656 -0.6727503 0.50422096 0.59954154 -0.5704234 -0.4165624 0.45891386 0.40617698 -0.7701897 -0.71972966 0.58876306 0.5631691 0.4791307 -0.5043819 0.61870587 0.59976023 -0.75037575 -0.4672895 0.48189896 0.7887305 0.631838 0.49927092 -0.46020994 -0.59535605 -0.4260217 0.6002698 0.5798574 -0.5003506 -0.66819805 0.4758429; -0.7635278 -0.79765874 0.5927659 -0.67323345 -0.59011704 0.9232798 0.6806691 -0.7722737 0.24954112 -0.47691453 0.7292301 0.55734265 -0.7973296 0.6319071 0.5643353 0.46403316 -0.5667178 0.4869039 -0.5089534 0.51646 0.6261618 -0.4823968 0.4132448 0.53040946 -0.53596455 -0.58073664 -0.75794464 0.7183513 -0.7482997 -0.64162284 0.48471162 0.3878786 -0.7968679 0.7533088 -0.41518798 0.54543746 -0.67953324 0.53136104 -0.3650438 0.67324126 0.6077975 0.4683958 -0.6811072 0.37189737 -0.6130032 0.705954 -0.4752223 0.94175005 0.47008935 0.56724435 0.5339736 -0.31196868 0.50225985 0.43934596 -0.49083695 -0.59833246 0.32312146 0.46583578 -0.73946583 -0.6612916 -0.42316365 0.4260782 0.5679399 0.29676107 0.05416598 -0.6866829 -0.54812944 -0.7724868 0.6730293 0.7639537 0.58771914 0.83017457 -0.61073405 0.72819096 0.29953957 0.5089585 0.7944153 -0.4781432 0.72468996 0.62047213; 0.43584007 0.6171031 0.78866917 -0.64301735 -0.59461665 0.5907846 -0.48222443 0.70444393 -0.44544706 -0.6466874 0.79303414 -0.8078301 0.56517524 0.49164298 0.49058008 -0.5343807 -0.6680139 0.4546844 0.46729812 0.6694347 0.42559683 0.67219734 -0.7804372 -0.48851582 0.008474898 -0.7151964 0.59945035 -0.7869985 -0.5786352 0.74876565 -0.68158376 -0.4188065 0.376024 0.42609963 -0.5232725 -0.6382024 -0.60857385 -0.65068275 -0.7821662 -0.15056989 0.7008253 0.73725456 -0.46793613 0.6792791 -0.68790185 -0.47182736 0.6568222 -0.5141616 -0.55292344 -0.42871982 -0.60808355 -0.6192975 0.4456208 0.53297323 0.76742667 -0.5491703 -0.6515002 -0.78635275 0.48920092 -0.578986 0.46282023 0.7803098 -0.5998121 -0.7407553 0.76960033 0.46367684 -0.4147679 -0.43818727 -0.56359124 -0.83908826 0.62863225 0.4465087 0.79100555 -0.62842363 -0.75885326 0.3842064 0.5969436 -0.76992905 -0.6276532 0.64181346; -0.41931462 -0.50347525 0.43662128 -0.53877366 -0.42192003 0.6475334 -0.54610175 -0.68407065 -0.707874 -0.41330287 0.4534691 0.6370768 -0.6084145 0.5422496 0.41834858 -0.4858058 -0.61068267 0.49404812 -0.58304405 0.5449178 0.5651296 -0.42126945 -0.74887496 -0.5400827 -0.66680574 -0.5256383 -0.7896188 0.38099068 -0.5941632 -0.7544568 0.662866 0.5520024 -0.47354078 0.49243417 -0.74957585 0.534052 -0.8042909 0.3829722 -0.55771095 0.69608617 0.6872614 0.4810468 -0.4952402 0.51320773 -0.5319001 0.51846445 -0.64137805 -0.6216441 -0.47853917 0.7013126 0.4171198 -0.7423114 0.44517267 0.716173 -0.53346974 -0.66378623 0.63396037 0.571621 -0.6359376 -0.57654667 0.7170677 0.6937257 0.43063378 -0.5764961 0.79695594 0.66280913 -0.42755875 -0.7523751 0.5740375 0.45824528 0.7653808 0.5679889 -0.6600266 -0.39892918 -0.7338869 0.47860673 0.63874024 -0.7721125 -0.50769305 0.6405524; -0.50341785 -0.5741994 0.7475998 -0.61182755 -0.58009046 0.4251022 0.7314844 -0.44720057 -0.69937795 -0.41387117 0.51052564 0.605272 -0.65021896 0.49420285 0.69904023 0.6613468 -0.6188171 0.7250551 -0.36679792 0.46156168 0.6738858 -0.5171578 -0.504981 0.4702923 -0.6906529 -0.7447087 -0.68912077 0.61966527 -0.69147843 -0.67762583 0.7273301 0.50633216 -0.3902991 0.7038457 -0.4688383 0.7580528 -0.62651175 0.5962519 -0.4750475 0.79329836 0.4602048 0.456078 -0.6512424 0.56211156 -0.6134342 0.7912211 -0.5686625 -0.4433906 0.8101657 0.6336726 0.58898526 -0.48709282 0.687722 0.59761775 -0.55054814 -0.61441785 0.5382807 0.47725952 -0.52439135 -0.4785911 0.64300126 0.6313356 0.45975623 -0.72873384 0.47065777 0.5373212 -0.62299246 -0.36765173 0.5864617 0.6208614 0.7733642 0.71596205 -0.828251 -0.5900053 -0.5234218 0.5949873 0.5842259 -0.68664086 -0.5629206 0.49760032; -0.6129857 -0.60902274 -0.6007805 0.7352106 0.6432188 -0.5065269 0.64958966 -0.6101922 0.55112404 0.7745826 -0.7758127 0.43248475 -0.6555435 -0.62419486 -0.5339222 0.74089205 0.5546377 -0.58818096 -0.40785277 -0.64561254 -0.50054616 -0.5086346 0.73569185 0.4641832 -0.28840736 0.5531428 -0.66958946 0.6996536 0.70246154 -0.5404496 0.49005157 0.4242419 -0.78270453 -0.82985395 0.5984519 0.6962613 0.73646116 0.568103 0.46747893 0.3067044 -0.5936871 -0.70810485 0.32264063 -0.4749845 0.79551685 0.3226805 -0.59199077 0.44690108 0.77769136 0.5245135 0.7535475 0.62394905 -0.53050727 -0.57892823 -0.7188678 0.64057726 0.3473115 0.5472579 -0.66931707 0.5204858 -0.5327402 -0.7214252 0.45090106 0.7370494 -0.61093676 -0.6379652 0.48680195 0.35885847 0.3841997 0.46746576 -0.4898895 -0.44168222 -0.729381 0.7226915 0.7015263 -0.5966966 -0.57119733 0.47316542 0.64610887 -0.59650517; 0.5276622 0.7313887 -0.50964737 0.43294218 0.7024277 -0.5676557 -0.73594433 0.66987485 0.66853166 0.7837129 -0.73606753 -0.5114655 0.6431317 -0.76325685 -0.44828033 -0.39836842 0.5007411 -0.46327537 0.58309025 -0.43011937 -0.7231032 0.45281985 0.51263314 -0.7628539 0.43540478 0.7099046 0.76890516 -0.68212354 0.6362102 0.43366522 -0.683355 -0.43183997 0.6859511 -0.43670517 0.45699224 -0.71200174 0.65937555 -0.77371037 0.5669794 -0.5718588 -0.4552362 -0.636082 0.6037064 -0.653579 0.6307349 -0.483986 0.716227 0.4090997 -0.75072175 -0.6051291 -0.5459727 0.41370517 -0.50683314 -0.42276198 0.59641784 0.67864525 -0.66172665 -0.71180123 0.7479608 0.4173256 -0.42645517 -0.41290537 -0.47393167 0.6432455 -0.6106913 -0.617622 0.40977263 0.5270514 -0.6381523 -0.7553051 -0.754803 -0.70723474 0.6791179 0.10114814 0.51441646 -0.584977 -0.70140016 0.51319355 0.73952216 -0.50710166; 0.6381346 0.44750765 -0.5070876 0.6072424 0.6382287 -0.7261588 0.8527303 0.43511894 0.39524397 0.73466927 -0.68905246 -0.6163304 0.5633098 -0.7241443 -0.57489336 0.4942847 0.4182806 -0.60035837 0.57032335 -0.76607174 -0.4570257 0.7353209 0.6077673 -0.49866042 0.43593812 0.73551774 0.73084337 -0.6598786 0.4789228 0.7344139 -0.5508784 -0.78084946 0.55774903 -0.48592117 0.53209877 -0.63241684 0.697622 -0.61677223 0.52918637 -0.5707885 -0.664398 -0.77231324 0.7328733 -0.70083356 0.6960856 -0.77617717 0.62726253 0.54536366 -0.4853964 -0.39586362 -0.38060576 0.68868047 -0.49735022 -0.61717427 0.66214526 0.7830284 -0.53514344 -0.68520755 0.5183145 0.7773627 -0.56989986 -0.4357719 -0.56213266 0.6944242 -0.5432401 -0.62031835 0.4437243 0.3914112 -0.7222774 -0.46828005 -0.42211306 -0.62706786 0.43219325 0.7456021 0.4674537 -0.50889814 -0.42408818 0.68323696 0.54554474 -0.4546006; 0.8027006 0.76076806 -0.072061285 -0.5587071 -0.4966839 0.5026108 -0.5864438 0.7140943 -0.48461872 0.24040689 0.20681779 -0.60986537 0.61171836 0.58503294 0.1486346 -0.59156686 0.32295522 -0.19950578 0.68790823 0.6156068 0.3327658 0.6993398 -0.6756465 -0.5803684 0.81486696 -0.32505083 0.8060372 -0.7044307 0.6075913 0.47919592 -0.4127669 -0.6160838 0.5966651 -0.11667822 -0.71336067 -0.36269388 -0.23398198 -0.48824805 -0.8451593 -0.6446565 0.42399693 0.4741765 0.5811077 0.7707022 -0.6078712 -0.44474214 0.7329145 -0.75728196 -0.6330472 -0.77357495 -0.5369962 -0.7507793 -0.40205878 0.7994947 0.68284255 0.8517776 0.012373684 -0.83298767 0.47945032 -0.50495094 0.8120461 -0.49680188 -0.42532364 -0.5262778 0.29126298 0.51714087 -0.17312093 -0.14174888 -0.26669672 -0.86864734 0.77324545 -0.6789608 0.46998823 -0.5792425 -0.47517642 0.12552024 -0.092939906 -0.37691554 -0.8748163 -0.82887536; -0.5128164 -0.5069479 0.7714898 -0.7945536 -0.78494316 0.6371469 -0.5960647 -0.49299172 -0.4573638 -0.67125577 0.5839332 0.566615 -0.5686024 0.6138126 0.4951599 0.75592935 -0.56747276 0.79163 -0.5056975 0.49238655 0.5895216 -0.51035345 -0.5289733 0.6852978 -0.4739292 -0.7201167 -0.59736544 0.77591074 -0.64716506 -0.67314476 0.68664646 0.7696891 -0.6068402 0.5640718 -0.5540804 0.69641495 -0.5256272 0.4794585 -0.4623377 0.5578862 0.4504513 0.6282109 -0.44418406 0.41600752 -0.4252825 0.63781387 -0.50572604 -0.5758716 0.5533585 0.6298054 0.72280806 -0.56148124 0.47004658 0.4250477 -0.6313473 -0.5571846 0.7510047 0.5137508 -0.67894495 -0.4302537 0.4729964 0.46351615 0.49424118 -0.6629782 0.744411 0.5284041 -0.51065165 -0.44097644 0.61379933 0.5844473 0.633696 0.49778575 -0.5625536 -0.4972228 -0.49534398 0.63941896 0.6171222 -0.47904685 -0.76749676 0.7607232; -0.6999056 -0.84446985 0.55421484 -0.5437124 -0.5591083 0.46928063 -0.068420134 -0.76021296 -0.6857392 -0.64742905 0.6048393 0.556702 -0.554523 0.4357698 0.69961923 0.34430858 -0.69854915 0.7041735 -0.7797224 0.38836512 0.5048774 -0.6062294 -0.7649761 0.4715392 -0.5299449 -0.40157992 -0.525974 0.7131849 -0.54695594 -0.57268995 0.6880285 0.4522891 -0.47393292 0.6888309 -0.46584645 0.47365424 -0.5334547 0.63849515 -0.6692764 0.6893082 0.73451495 0.5384464 -0.79824626 0.7180079 -0.47883746 0.45410365 -0.70654666 -0.3413365 0.40497124 0.7773563 0.8277221 -0.5022354 0.64690834 0.38715613 -0.614363 -0.7110366 0.55508083 0.71035886 -0.49925244 -0.56296843 0.6519067 0.6321656 0.74829113 -0.8050636 0.6338306 0.5560155 -0.46500048 -0.791709 0.3100767 0.6712613 0.5833657 0.641558 -0.63209087 -0.4606203 -0.56099474 0.57806736 0.7308471 -0.4889383 -0.68085784 0.49697012; -0.86739135 -0.72812104 0.5422437 -0.65337044 -0.7588868 0.5051157 -0.6570101 0.58532596 -0.71034616 -0.44363257 0.7622602 0.46702582 -0.98913 0.5608635 0.7981402 -0.67427576 -0.48623088 0.51574665 -0.5284967 0.6745539 0.6490063 -0.95783967 -0.57340926 -0.6006598 -0.5203131 -0.6077448 -0.457504 -0.59789544 -0.65090716 -0.55841607 -0.233261 -0.6640463 -0.884409 0.3867706 -0.5087803 -0.58273333 -0.85001147 -0.88493925 -0.39310995 0.4060018 0.6739608 0.71069825 -0.7616041 0.6582354 -0.5454604 0.57966495 0.59456766 -0.6323188 -0.7842285 0.91547185 -0.45136175 -0.47155073 0.43753615 0.60495055 -0.96273166 -0.7146282 -0.020733278 0.72740614 0.37302846 -0.47469738 0.75850064 0.5829076 0.41145465 -0.75971985 0.7489652 0.58032775 -0.81895936 -0.58648455 -0.0018947556 0.7813242 0.42202932 0.7145326 -0.45020393 -0.58925503 -0.6888727 0.28209493 0.59229404 -0.38688204 -0.40443462 0.57773554; 0.45463365 0.51858836 0.7190798 -0.6493385 -0.57268596 0.46899235 -0.4706647 0.7966363 -0.7274384 -0.4586034 0.48385224 -0.6394871 0.61104125 0.5768983 0.76229155 -0.60204124 -0.6828854 0.508819 0.65898806 0.51286376 0.5134243 0.5854703 -0.57635826 -0.76748514 0.63867533 -0.42073235 0.6962394 -0.5478504 -0.57227856 0.45718494 -0.7302553 -0.44646704 0.5778389 0.4132536 -0.68353176 -0.73716956 -0.5994176 -0.43289378 -0.63660824 -0.74049145 0.49643815 0.6069123 0.47220105 0.7248973 -0.7417388 -0.42845234 0.78472733 -0.60386306 -0.50215805 -0.548009 -0.6015022 -0.5032795 0.5725127 0.7676177 0.6442501 -0.5803824 -0.3543963 -0.7711158 0.75168973 -0.4185925 0.6048905 0.60161996 -0.48479542 -0.74628526 0.764221 0.7512935 -0.6967239 -0.66568196 -0.48945287 -0.67671025 0.63075304 -0.6748139 0.53000283 -0.70690155 -0.66740537 0.5173637 0.4824574 -0.6843239 -0.7704141 -0.74077344; 0.7333756 0.5522841 -0.47766072 0.75479543 0.47091007 -0.6573475 0.66311234 0.80965674 0.80951643 0.6083492 -0.5495337 -0.458923 0.80005264 -0.59511703 -0.7925271 0.5778002 0.47244266 -0.5826098 0.6899539 -0.60654 -0.3834829 0.49617404 0.52629566 0.9393068 0.70673037 0.68982965 0.59668946 -0.6131249 0.7652262 0.54862845 -0.5833017 -0.30027536 0.64116436 -0.4136539 0.7460023 -0.08659372 0.50416327 -0.31818774 0.67921406 -0.38869855 -0.7396152 -0.5844496 0.7960336 -0.41218764 0.46860895 -0.66323674 0.8402418 0.5991683 0.9392984 -0.59225136 0.6538542 0.47817427 -0.48729464 -0.4460736 0.4962857 0.44523498 -0.5581152 -0.76545274 0.38728 0.7648253 -0.77839684 -0.4433291 -0.49047545 0.5083733 -0.79140943 -0.6120137 0.54078364 0.544387 -0.41062385 -0.55938184 -0.40075245 -0.4499461 0.3587032 0.7451938 0.79041266 -0.37627771 -0.650134 0.5242593 0.5025037 -0.7611663; -0.60558736 -0.800791 0.4357134 -0.43159768 -0.39159656 0.73790854 -0.5521079 0.5963401 -0.49622825 -0.562398 0.70949244 0.71416944 -0.6514358 0.6127576 0.7492047 -0.5807349 -0.64463764 0.4487329 -0.46411878 0.41322607 0.48128724 -0.6171678 -0.64404285 -0.73390746 -0.80257404 -0.7081047 -0.56576335 -0.6177539 -0.6096851 -0.46651042 0.4371312 -0.46796203 -0.79897594 0.56019765 -0.74409425 -0.718966 -0.6585543 -0.7296364 -0.756633 0.3877651 0.7170317 0.6446008 -0.56999326 0.535357 -0.6352582 0.6231553 0.68030137 -0.7424043 -0.81529653 0.40635845 -0.41083312 -0.7633088 0.43820974 0.7262539 -0.5479891 -0.76094526 0.4128316 0.47209358 0.3488532 -0.61806387 0.50697595 0.5778283 0.44908723 -0.5864409 0.7916726 0.7240978 -0.8275918 -0.6273862 0.461525 0.6203716 0.4043543 0.5418112 -0.70565355 -0.6107007 -0.5157857 0.6142333 0.78397405 -0.6974959 -0.3836592 0.5754103; -0.49761626 -0.644016 0.4373987 -0.71577275 -0.5267257 0.7774517 0.580658 -0.7289015 -0.63750917 -0.674499 0.6187033 0.49214965 -0.5073567 0.6332033 0.6523581 0.737932 -0.74469703 0.664667 -0.53573954 0.7244386 0.7250427 -0.6822531 -0.6585716 0.41956264 -0.5165095 -0.75893945 -0.5634597 0.59428644 -0.7377716 -0.59096634 0.44689685 0.43611366 -0.7358751 0.4578488 -0.5999531 0.67751116 -0.5949515 0.50373024 -0.65412396 0.5883502 0.40574193 0.5346245 -0.79593277 0.6493244 -0.3948561 0.76457304 -0.48623872 -0.37579766 0.7033557 0.7797085 0.73310614 -0.50640386 0.47669396 0.47220328 -0.51119834 -0.54878783 0.41324118 0.64853454 -0.43096074 -0.51675063 0.5024132 0.64243567 0.4230204 -0.6503357 0.2944812 0.44972306 -0.65467936 -0.65999734 0.47001517 0.7381917 0.6701808 0.47640103 -0.49336457 -0.557451 -0.5040267 0.68031585 0.7398775 -0.7320433 -0.43429676 0.76226854; 0.8241762 0.48229614 -0.7582168 0.5056871 0.55310506 -0.737999 0.4758779 0.4618438 0.43689662 0.7375509 -0.7850786 -0.798917 0.7435277 -0.5122782 -0.6238638 0.6165208 0.7410296 -0.6982558 0.52572423 -0.77978253 -0.65317714 0.67727596 0.70747185 0.4044584 0.42391315 0.763983 0.64734364 -0.3328418 0.5553573 0.5975917 -0.38828504 0.75972337 0.46240208 -0.6784813 0.71249825 0.39217642 0.7771795 0.43682826 0.53368807 -0.7329328 -0.67156833 -0.5711158 0.53741205 -0.6362921 0.7706906 -0.6861374 -0.42801863 0.4991686 0.4097246 -0.7653897 0.73080045 0.7788459 -0.49534974 -0.6603637 0.46106383 0.6504957 -0.27395526 -0.64217675 -0.43908614 0.63125205 -0.4892471 -0.7766902 -0.40976766 0.43872538 -0.60165775 -0.4589038 0.49982446 0.7006224 -0.4574576 -0.559815 -0.6868999 -0.4956246 0.575373 0.42643076 0.5402817 -0.6101045 -0.7076559 0.7830861 0.51788056 -0.74632293; -0.43777263 -0.50262433 -0.5319382 0.47829613 0.47101012 -0.76054955 0.49074343 -0.7713413 0.45968816 0.71121716 -0.6162841 -0.33954412 -0.6616109 -0.43973362 -0.7716088 0.6305946 0.78950864 -0.5387843 -0.2564627 -0.71355546 -0.76074314 -0.6084219 0.48890284 0.75407755 0.4834664 0.69763684 -0.604253 0.7208169 0.7544499 -0.48523146 0.5987712 0.5305053 -0.070453115 -0.7589491 0.7208122 0.44448698 0.45686126 0.6443238 0.5136335 -0.6975976 -0.6004104 -0.7701688 0.64427453 -0.6135057 0.4527778 -0.48755747 -0.6309479 0.4917524 0.76965714 0.47492906 0.6622227 0.75247294 -0.7627857 -0.8048636 -0.64055926 0.533059 0.2453644 0.6345169 -0.49104184 0.6531053 -0.69916767 -0.76216865 -0.7146775 0.7647965 -0.49826658 -0.60301155 0.52619225 0.54987055 0.5982409 -0.42458487 -0.56324315 -0.5553439 -0.46767327 0.6587434 0.4030543 -0.46831834 -0.49313408 0.6088545 0.44630995 -0.6543953; -0.71100485 -0.48410696 0.41835034 -0.79017365 -0.7045523 0.43557715 0.49266478 -0.4175791 -0.56382096 -0.43419918 0.57660687 0.63549197 -0.39135027 0.42282933 0.42320904 0.7147409 -0.5175623 0.53872895 -0.50430846 0.48134243 0.418 -0.55564046 -0.7012735 0.7105775 -0.68973565 -0.5356672 -0.46178988 0.49539173 -0.7321345 -0.668874 0.43885705 0.6793946 -0.7528879 0.5033412 -0.56375396 0.75284266 -0.49370167 0.7519652 -0.79428655 0.57819825 0.44550213 0.46644595 -0.4991896 0.57939214 -0.5890323 0.7427372 -0.40483615 -0.5566618 0.5650211 0.41647822 0.5159552 -0.6554812 0.53764313 0.50328875 -0.41174304 -0.4307007 0.44054538 0.6936731 -0.5056034 -0.5327624 0.52312386 0.5251979 0.77994424 -0.69082123 0.78609717 0.5554079 -0.7185677 -0.65691113 0.48657554 0.552025 0.63625973 0.51523274 -0.51206374 0.10829073 -0.45375666 0.5641611 0.37646863 -0.67143846 -0.48641267 0.52351946; 0.71903485 0.66512877 -0.41256458 0.6836859 0.56204814 -0.584762 0.39404023 0.55390275 0.4511641 0.5760694 -0.590838 -0.6851633 0.5192513 -0.7216914 -0.7452323 0.62504846 0.47007123 -0.7657492 0.6155191 -0.45941746 -0.6275581 0.5782219 0.55472237 -0.5915332 0.6460758 0.5771 0.75077224 -0.40163398 0.5156109 0.4366256 -0.7381829 -0.687692 0.51305497 -0.6586426 0.4696927 -0.565887 0.6813717 -0.386297 0.76948726 -0.6654637 -0.66045463 -0.6553263 0.5919858 -0.5939388 0.4059816 -0.49821234 0.68048936 0.44450805 -0.32041523 -0.54530287 -0.6827322 0.4748162 -0.70591164 -0.756945 0.580399 0.578466 -0.59403974 -0.7375743 0.5358986 0.7060749 -0.53739756 -0.7469001 -0.724645 0.48899043 -0.49877784 -0.6376852 0.51946366 0.6493971 -0.38693658 -0.7164962 -0.42847723 -0.7475887 0.735128 0.57029235 0.65874296 -0.4203176 -0.6935578 0.5135094 0.45555738 -0.48941293; -0.4501933 -0.5912045 0.79689944 -0.56428903 -0.73070467 0.72479296 0.6545832 -0.7376023 -0.47028577 -0.6705004 0.7225198 0.7473156 -0.5154097 0.6565867 0.7252178 0.57177806 -0.60426366 0.8089066 -0.48493809 0.68733674 0.44262058 -0.5408705 -0.5538276 0.7304897 -0.671625 -0.6922807 -0.4451558 0.5094555 -0.47724286 -0.49341905 0.5871675 0.72584736 -0.50207186 0.5297196 -0.48760846 0.6155012 -0.65795064 0.7163661 -0.75121087 0.53874505 0.44531298 0.756293 -0.6264379 0.5685258 -0.6002325 0.60241586 -0.7309729 -0.69974923 0.7740629 0.72309166 0.5733682 -0.7365852 0.55562186 0.51900804 -0.6868667 -0.72231656 0.5894277 0.50712514 -0.39235872 -0.6001344 0.66188437 0.7836819 0.55110383 -0.46644223 0.31778663 0.4443598 -0.73062205 -0.5864515 0.8033541 0.46638957 0.49935257 0.6261373 -0.8210364 -0.6989023 -0.51243037 0.718238 0.6560988 -0.54001474 -0.46844214 0.48634472; -0.71159047 -0.7455506 0.78854614 -0.5069304 -0.46868026 0.6691943 -0.80058694 -0.68556184 -0.65259534 -0.48215055 0.5205933 0.6701816 -0.42859152 0.6161729 0.66807663 0.7845537 -0.48476928 0.41331053 -0.57045174 0.6966514 0.7136076 -0.4857399 -0.42218202 0.7884933 -0.7080631 -0.45656252 -0.5936784 0.59069693 -0.6468368 -0.7942865 0.5908144 0.7244348 -0.49864393 0.51793754 -0.46003073 0.5061351 -0.4396151 0.7269011 -0.5223418 0.6965387 0.60362405 0.543571 -0.39867273 0.633078 -0.4518105 0.42952293 -0.52150846 -0.76733744 0.77544326 0.72352815 0.59827393 -0.7043379 0.5108114 0.44201484 -0.5277704 -0.5102961 0.6845325 0.7197528 -0.72052366 -0.6924674 0.75447065 0.43401617 0.62111354 -0.3987621 0.4623478 0.5472881 -0.6379618 -0.4882309 0.49637347 0.40876904 0.6096992 0.5525065 -0.68853164 -0.54078126 -0.5726639 0.5790919 0.771009 -0.6623815 -0.56119037 0.5759144; 0.74981725 0.6528607 -0.6717138 0.5187988 0.50619024 -0.47566104 -0.48363864 0.6883137 0.7349826 0.50296545 -0.6318663 -0.4639231 0.7559797 -0.6519052 -0.49079126 -0.4132545 0.6810545 -0.43311775 0.7898476 -0.552157 -0.6670986 0.6163825 0.46368784 -0.3978645 0.694407 0.5693451 0.69637656 -0.50262535 0.5827629 0.5163253 -0.54688877 -0.42854935 0.72453743 -0.6822158 0.4344931 -0.4060524 0.7841226 -0.41959706 0.639192 -0.6926176 -0.47468802 -0.43930906 0.73596334 -0.76995295 0.50896764 -0.6341286 0.5993981 0.41286385 -0.49802983 -0.44335997 -0.5925507 0.58430713 -0.41585746 -0.53830373 0.7944081 0.68453264 -0.46492758 -0.7656779 0.7853748 0.61996114 -0.6728971 -0.4963043 -0.5726978 0.6213569 -0.7489962 -0.5605034 0.7623463 0.5371211 -0.5137571 -0.70007724 -0.63374686 -0.62317026 0.4392905 0.44803867 0.6400378 -0.61950445 -0.44673425 0.5313866 0.72301906 -0.65822697; 0.63228685 0.774879 -0.6927164 0.41940644 0.7533002 -0.6615142 -0.66997206 0.44112128 1.2164266 0.64756566 -0.48822036 -0.58611333 0.57410663 -0.6843389 -0.7242685 -0.46188262 0.602424 -0.4776276 0.4112001 -0.44611076 -0.7312896 0.68724036 0.68071336 -0.55134046 0.54699737 0.5409597 0.72296566 -0.601358 0.7954762 0.42274672 -0.4767313 -0.68087715 0.5422293 -0.48530412 0.6950548 -0.7400936 0.6523895 -0.6915945 0.4554348 -0.6681702 -0.531923 -0.6868922 0.6176052 -0.40115446 0.6787152 -0.42601463 0.7533128 0.46683306 -0.6182179 -0.5077467 -0.64411145 0.42460757 -0.62815017 -0.41391504 0.47358924 0.6652918 -0.48427224 -0.59675884 0.55357033 0.64179873 -0.5808686 -0.39605373 -0.71418405 0.76957214 -0.6616129 -0.8248014 0.80232495 0.7652111 -0.45442286 -0.49824095 -0.49149263 -0.43679714 0.408869 -0.8908224 0.62550133 -0.3909944 -0.5445581 0.63310975 0.44295564 -0.5126941; -0.5024105 -0.5073596 0.4890717 -0.43970954 -0.67480505 0.65631753 0.5725662 -0.788024 -0.9158491 -0.5758178 0.6661043 0.5388902 -0.7680908 0.7516496 0.4832847 0.7421114 -0.6093086 0.5106384 -0.5559404 0.7082763 0.7205918 -0.68355423 -0.6616792 0.5200341 -0.5127453 -0.5571635 -0.46484473 0.5174598 -0.70393795 -0.6663795 0.64606583 0.421253 -0.47202942 0.6095169 -0.45259106 0.5501018 -0.5297926 0.76531535 -0.4649709 0.6168738 0.432871 0.42528847 -0.7848119 0.60954034 -0.6035453 0.4089914 -0.6671703 -0.4638485 0.69441634 0.6644402 0.4330903 -0.47286883 0.7614022 0.7218335 -0.7385456 -0.6726158 0.4021602 0.53258437 -0.77024275 -0.454603 0.5133548 0.5692202 0.5064091 -0.589142 0.6713607 0.812936 -0.53136015 -0.60240674 0.6688014 0.7695136 0.45036045 0.7306456 -0.641405 0.5293919 -0.6243752 0.46667823 0.62592787 -0.70883113 -0.6364346 0.75387347; -0.53910136 -0.5889057 0.40651298 -0.5081876 -0.66568154 0.6283839 0.64678615 -0.42499202 -0.45782405 -0.6594512 0.55358523 0.41782698 -0.6451855 0.40978906 0.652464 0.74400544 -0.41991165 0.5538942 -0.7685018 0.46529317 0.61760455 -0.5281479 -0.63275445 0.4360498 -0.7102273 -0.416484 -0.5137765 0.4486309 -0.67874885 -0.77779365 0.52862376 0.5219322 -0.69347626 0.7844193 -0.4849888 0.41935095 -0.7376634 0.434766 -0.75887245 0.75822574 0.7706096 0.5438444 -0.6973253 0.72764444 -0.58648634 0.48290378 -0.6391557 -0.49184296 0.5874106 0.4621827 0.7751684 -0.6113342 0.639583 0.4592338 -0.7807583 -0.5607993 0.67463034 0.7336856 -0.46503127 -0.5359557 0.4509703 0.4887134 0.5534068 -0.57480586 0.5903666 0.6017525 -0.7594454 -0.712977 0.4453975 0.7221032 0.43009675 0.5008539 -0.7112888 -0.6058582 -0.54445046 0.41654804 0.40824747 -0.42086884 -0.44163576 0.6876352; 0.64629924 -0.39053544 0.46661285 -0.44967777 -0.66151756 0.5157149 -0.6722136 0.39292294 -0.51064414 -0.67542934 0.4200027 0.66361964 -0.3920282 0.6581812 0.42712778 -0.6119838 -0.37823758 0.6040099 -0.5191513 0.7315153 0.48531267 0.64784473 -0.44855645 -0.47103402 -0.43700707 -0.59797513 -0.7422799 -0.80363786 -0.42486972 -0.6739384 -0.6003485 -0.67384726 -0.7891762 0.60483295 -0.6307585 -0.7095619 -0.7163535 -0.4451214 -0.40748504 0.6462768 0.769526 0.62839174 -0.6472224 0.6594817 -0.4621346 0.39690468 0.39731276 -0.57902694 -0.65987414 -0.62983894 -0.79172915 -0.70029557 0.6591229 0.5099701 0.50077057 -0.41141078 0.15598975 -0.74621046 0.4799429 -0.69262 0.76337206 0.42086455 0.39080825 -0.5847643 0.8055479 0.45566085 -0.74795765 -0.61516005 -0.1633372 0.6188167 0.48069045 0.5222642 0.5511053 -0.5113321 -0.56048805 0.432381 0.65168405 -0.6493815 -0.6371662 0.5206958; 0.7844099 0.5712353 -0.43651968 0.413534 0.6347547 -0.68639237 0.5573293 0.6042448 0.84514797 0.4498225 -0.780102 -0.461528 0.7404683 -0.658486 -0.5279961 0.8370412 0.60186803 -0.5340597 0.6558839 -0.58797616 -0.53180695 0.4824076 0.8042436 -0.31999874 0.56943214 0.5142177 0.6453141 -0.39047033 0.57524985 0.62756956 -0.55351394 -0.63454926 0.6068211 -0.47380072 0.52143055 -0.5201775 0.7980821 -0.3551699 0.7235406 -0.49261865 -0.655708 -0.62044895 0.66070235 -0.4385682 0.40398496 -0.61420476 0.74065673 0.5243216 -0.06555057 -0.51713574 -0.6916384 0.45434052 -0.6089 -0.6548011 0.56222016 0.6167076 -0.31404775 -0.67971617 0.8616787 0.49334347 -0.7064817 -0.49911124 -0.49422362 0.61248845 -0.6466967 -0.78061485 0.5712693 0.673818 -0.33049655 -0.5531344 -0.50371087 -0.42740548 0.6023667 0.4635965 0.6608111 -0.45623973 -0.49854037 0.67053586 0.644767 -0.7396131; 0.37864378 0.52961046 0.7253247 -0.68129945 -0.63841575 0.54158187 -0.69721884 0.6613828 -0.56024384 -0.57711065 0.54603297 0.47162113 0.5082136 0.67413914 0.6393729 -0.56140167 -0.61138964 0.69733256 0.18473956 0.49508077 0.8121886 0.65319407 -0.43492058 -0.68750316 -0.30048433 -0.69725883 0.38693 -0.49913943 -0.6531842 0.305428 -0.18176764 -0.46667865 0.32753003 0.52317786 -0.71813434 -0.6527913 -0.33951947 -0.584909 -0.66125304 0.54845315 0.70261526 0.5341169 -0.7263747 0.7866742 -0.7329086 0.5868213 0.43158352 -0.5970217 -0.43528867 -0.65438145 -0.51619714 -0.46712646 0.86801285 0.82672435 0.46007684 -0.6506474 0.14216153 -0.5302797 0.7992232 -0.5421513 0.69846946 0.71649474 0.83165926 -0.7015694 0.12733233 0.35795507 -0.32797897 -0.35467988 -0.2654987 0.4410457 0.7166681 0.54123163 0.2874903 -0.4824482 -0.48611844 0.48498088 0.41421542 -0.8083316 -0.81344265 0.545332; 0.6116454 0.68460584 -0.6926584 0.5890855 0.69294214 -0.57705736 -0.65513843 0.40136728 0.50254357 0.69827497 -0.4042498 -0.633298 0.38743111 -0.43976238 -0.67591435 -0.63737255 0.49913043 -0.7482719 0.72988397 -0.4921215 -0.47072113 0.5274264 0.6499473 -0.7512254 0.47736692 0.5516568 0.47368976 -0.8200956 0.45077828 0.4003121 -0.49829495 -0.58159477 0.62239367 -0.6379856 0.617999 -0.7284255 0.6047329 -0.6119796 0.78181994 -0.74556565 -0.54055876 -0.5768576 0.576627 -0.62908524 0.5552675 -0.66609275 0.75351095 0.5615524 -0.82029825 -0.59679246 -0.7214237 0.6717124 -0.6206222 -0.53685766 0.5260874 0.5076932 -0.74443066 -0.394539 0.40483207 0.7614483 -0.7738992 -0.5037896 -0.603928 0.6475223 -0.44903818 -0.6108738 0.74287736 0.6839824 -0.6555958 -0.6440451 -0.71069646 -0.7375263 0.70616686 0.7967182 0.70127934 -0.53993785 -0.713403 0.67525715 0.5998974 -0.68955684; 0.70141053 0.68885076 -0.42947692 0.69771004 0.41843817 -0.72798926 0.763617 0.6831823 0.5174946 0.76284254 -0.6722299 -0.66952 0.76458216 -0.6301719 -0.49315125 0.61176085 0.5211253 -0.5043037 0.74552345 -0.76161814 -0.77311164 0.5801295 0.41004983 0.42355597 0.5573819 0.7673781 0.44112262 -0.5297756 0.7052607 0.71859705 -0.634965 0.2594227 0.6033303 -0.76495206 0.54762053 0.018075617 0.7962476 -0.46832 0.7572647 -0.5387283 -0.5862912 -0.4694935 0.43001005 -0.43178204 0.4940596 -0.77006435 0.2367612 0.5923642 0.5627369 -0.788119 0.33682752 0.5715376 -0.73727924 -0.6260108 0.599738 0.49819875 -0.45368582 -0.6766541 -0.67655593 0.40823144 -0.46994343 -0.7952294 -0.5599024 0.6517401 -0.41861495 -0.74449885 0.4381763 0.81126684 -0.6182731 -0.7218108 -0.574825 -0.65927017 0.5646674 0.6945097 0.7909559 -0.6831456 -0.6088269 0.63367945 0.54805803 -0.52295345; -0.68078244 -0.655805 0.43487218 -0.69245285 -0.7274829 0.7344377 0.71248597 -0.65487903 0.29513484 -0.68948156 0.65729463 0.48193145 -0.5969173 0.70573723 0.5576277 0.6668144 -0.5706943 0.69145536 -0.50375444 0.6386226 0.45293003 -0.5954285 -0.42960593 0.6580756 -0.7450281 -0.7779817 -0.43403617 0.64271176 -0.5494113 -0.78550446 0.6281396 0.43449098 -0.6562676 0.45494306 -0.72866994 0.43315232 -0.7197295 0.4512899 -0.5259256 0.6445709 0.59824085 0.4397018 -0.58581483 0.7126454 -0.509425 0.5460229 -0.6638655 -0.6148809 0.49214908 0.6797154 0.6541697 -0.53557587 0.76832724 0.7174014 -0.79073584 -0.6245804 0.42585033 0.43351394 -0.7356072 -0.4420103 0.37716693 0.5551358 0.6131578 -0.7869037 0.40257156 -0.50877666 -0.42765895 -0.6549322 0.6211124 0.7929753 0.61340886 0.6007302 -0.5034913 0.54572755 -0.39107272 0.4509592 0.7640753 -0.6389343 -0.43004388 0.80719846; -0.49903262 -0.6980095 0.661753 -0.43259218 -0.5355339 0.5047931 0.54649204 -0.56499195 -0.86655676 -0.6535259 0.75665796 0.66989946 -0.7952951 0.55670494 0.83484966 0.3928491 -0.6206883 0.5190701 -0.69485074 0.6710051 0.6239659 -0.8066823 -0.6592541 0.572421 -0.70092416 -0.5015337 -0.5230719 0.615382 -0.50997293 -0.58965445 0.5086758 0.5479366 -0.8074696 0.6633511 -0.7103727 0.3903588 -0.8458894 0.6200088 -0.5563468 0.37984732 0.4170226 0.46727788 -0.73911303 0.36588797 -0.7480891 0.4426821 -0.7109788 -0.70092076 0.3833409 0.5527227 0.64893186 -0.7121687 0.63957167 0.52624995 -0.5531091 -0.6482024 0.3989612 0.6224443 -0.77334535 -0.6614213 0.479297 0.6637245 0.630748 -0.8134577 0.8147142 0.6347183 -0.64118934 -0.6735298 0.615961 0.71177167 0.4878024 0.5878991 -0.43967894 -0.2668762 -0.5329896 0.7042581 0.47597268 -0.5695806 -0.6714411 0.707426; 0.74975693 0.54219705 -0.44797513 0.40127108 0.5462304 -0.6973093 -0.72320557 0.7793959 0.49242043 0.77931887 -0.7575376 -0.77191925 0.500007 -0.5816407 -0.79363465 -0.5506171 0.7552953 -0.51385427 0.5392674 -0.63766617 -0.6102917 0.6030116 0.80033106 -0.4054269 0.6667789 0.6857096 0.71948105 -0.39857224 0.7940466 0.55935055 -0.3840547 -0.41967377 0.76462555 -0.5827764 0.53902423 -0.46804726 0.56745994 -0.40844384 0.63694525 -0.48867938 -0.53070563 -0.44276717 0.46558827 -0.4703204 0.5409864 -0.4444428 0.5000616 0.73382014 -0.42169487 -0.66626614 -0.5165068 0.693243 -0.4731315 -0.7546186 0.5275883 0.59336907 -0.4898106 -0.4630236 0.7000529 0.50952214 -0.5320878 -0.7435019 -0.6138907 0.5071605 -0.67724884 -0.6849009 0.81176704 0.4921867 -0.4752411 -0.78985405 -0.418794 -0.63038635 0.4933501 0.6812423 0.74114835 -0.75395066 -0.5455129 0.55943036 0.7351008 -0.7871139; 0.5783058 0.68684393 -0.7284 0.54068714 0.48701715 -0.47229433 0.7151141 0.66811657 0.60047597 0.422109 -0.70306593 -0.67153007 0.3992774 -0.71929675 -0.56017196 0.15192035 0.7616867 -0.5667959 0.5712395 -0.5123263 -0.46097004 0.6887233 0.62668437 -0.93166333 0.6999439 0.6486171 0.6996566 -0.8391914 0.673413 0.47364986 -0.6808869 -0.82175815 0.6162258 -0.5997239 0.6205074 -0.6802565 0.43621102 -0.66251177 0.5578411 -0.5755291 -0.4953056 -0.6679958 0.73936254 -0.48308718 0.68676996 -0.83595127 0.6736806 0.6656362 -0.9161271 -0.58822846 -0.5425128 0.68577373 -0.6697366 -0.5174456 0.5845833 0.53384507 -0.748735 -0.69831204 0.08081598 0.8090369 -0.5079993 -0.6549142 -0.4521705 0.57540035 -0.21308276 -0.63429564 0.5644344 0.6653552 -0.7245888 -0.73555017 -0.5726709 -0.68546975 0.6307364 0.55533 0.65578 -0.7153842 -0.6723171 0.6259544 0.5879158 -0.634917; 0.5735787 0.75338894 -0.45120475 0.62430143 0.5410475 -0.6086554 -0.59105957 0.502723 0.6140783 0.5683596 -0.635975 -0.4650289 0.5526735 -0.7521365 -0.6330608 -0.7007377 0.45162174 -0.5207318 0.7531738 -0.5355474 -0.7591892 0.792763 0.6235357 -0.62540245 0.6120806 0.7647589 0.69821465 -0.49257192 0.5403923 0.46330604 -0.7255838 -0.6641756 0.4813853 -0.46071708 0.7381337 -0.45607066 0.70833755 -0.67095804 0.48512742 -0.40102428 -0.5176605 -0.660714 0.57104564 -0.52348846 0.7551289 -0.41289973 0.6791241 0.3900235 -0.4651576 -0.67005247 -0.6249328 0.6141212 -0.6303894 -0.42168948 0.717524 0.5832143 -0.50091374 -0.5630651 0.7780417 0.46581316 -0.679213 -0.42279133 -0.57032526 0.4339256 -0.69039756 -0.8723989 0.6759625 0.51515836 -0.75275856 -0.7730168 -0.4810849 -0.45252895 0.67280287 -0.90519065 0.85142314 -0.498574 -0.6939784 0.68072635 0.560252 -0.48092097; 0.6765846 0.503559 -0.6009858 0.6337337 0.6510721 -0.6561022 -0.7498077 0.45283675 0.5649687 0.6845518 -0.38239962 -0.5357255 0.6066169 -0.70016056 -0.4192138 -0.559116 0.68217224 -0.7806073 0.7435457 -0.6544147 -0.7023533 0.61210227 0.7008284 -0.81972754 0.5356413 0.62126815 0.59571457 -0.6369937 0.6045739 0.5783938 -0.59744596 -0.72746515 0.42270702 -0.7772872 0.61029136 -0.8054251 0.52609134 -0.51405114 0.5187282 -0.44011745 -0.49114814 -0.4472626 0.7276697 -0.6358903 0.5681067 -0.6442064 0.7157391 0.5364145 -0.45276797 -0.7150735 -0.3881044 0.64685285 -0.5294315 -0.75689864 0.5863848 0.68485224 -0.7377205 -0.44755444 0.701899 0.57058465 -0.78441215 -0.5835777 -0.5938434 0.5300242 -0.23162419 -0.70056427 0.46103767 0.40675494 -0.61126137 -0.74856836 -0.45539206 -0.6294933 0.46212956 0.80716914 0.6342404 -0.6232786 -0.50916916 0.7313112 0.5625531 -0.74318796; -0.59958667 -0.5159818 0.7062519 -0.48936012 -0.6516034 0.35365856 0.53751373 -0.38175184 0.5079371 -0.5231718 0.5942159 0.5340901 -0.5602785 0.55555815 0.41578215 0.52659714 -0.5938863 0.4271244 -0.4223002 0.7595124 0.6155328 -0.5626594 -0.6147143 0.54373217 -0.7149122 -0.76859474 -0.5358657 0.5781288 -0.5123711 -0.76896906 0.75239813 0.66283983 -0.4083255 0.76659256 -0.55202574 0.7443759 -0.6533456 0.77424943 -0.5615217 0.49137145 0.41753677 0.51448774 -0.6073614 0.67786753 -0.44717652 0.7010221 -0.44176924 -0.45194954 0.7163677 0.6810801 0.46663737 -0.4528549 0.41915688 0.5508962 -0.6358659 -0.5648372 0.5576567 0.503679 -0.55885845 -0.5201433 0.5188769 0.43373632 0.49624863 -0.36794302 0.1400703 -0.72389024 -0.48311064 -0.342975 0.4923666 0.62049955 0.55374616 0.3882233 -0.62214464 0.49850065 0.6859856 0.5804209 0.64099175 -0.7090401 -0.7814572 0.6698526; 0.5299837 0.56642985 -0.40561095 0.6921575 0.48478746 -0.70937485 -0.41380998 0.4368666 0.7735852 0.43124458 -0.7484837 -0.7113257 0.48147434 -0.5478279 -0.6681218 -0.68458724 0.753028 -0.555464 0.6769094 -0.46138066 -0.49578294 0.7009984 0.64777637 -0.76897 0.43374497 0.56622547 0.7666902 -0.777295 0.6260578 0.6577107 -0.46556193 -0.614925 0.43148842 -0.50237346 0.71412265 -0.75063586 0.7581073 -0.5484973 0.40409398 -0.46937668 -0.77125156 -0.5537077 0.48075414 -0.4624618 0.41073078 -0.46725366 0.64334506 0.4480413 -0.5509045 -0.43698412 -0.706276 0.4778959 -0.41188306 -0.45144093 0.454624 0.53484833 -0.6678507 -0.79209375 0.5000053 0.45878962 -0.47130874 -0.75769544 -0.65757096 0.45741513 -0.5649704 -0.74955404 0.644898 0.5781814 -0.53625536 -0.7830708 -0.40912518 -0.6357512 0.62121314 0.60223174 0.71734434 -0.43564042 -0.7892189 0.42704573 0.59550864 -0.7842404; -0.47545692 -0.675191 0.77650946 -0.6648748 -0.68153757 0.6824343 0.7412084 -0.6401504 -0.32880184 -0.6873274 0.5510297 0.6062595 -0.4545086 0.54814017 0.6419363 0.4425336 -0.65722895 0.5540775 -0.4304072 0.40866718 0.73112136 -0.8158617 -0.6091133 0.44091415 -0.4621145 -0.5417802 -0.4819957 0.5208852 -0.4691521 -0.70000774 0.7119048 0.39476678 -0.47989836 0.51270175 -0.49180177 0.63780606 -0.7859939 0.49257857 -0.5232233 0.5846167 0.56651646 0.6464839 -0.7717056 0.44585806 -0.7588069 0.712179 -0.6774257 -0.3593928 0.5967707 0.6587388 0.80760515 -0.4084799 0.5194917 0.49587408 -0.67249763 -0.80154294 0.6423838 0.7006757 -0.6264074 -0.76192254 0.5750777 0.57166845 0.6520888 -0.8382208 0.5655503 -0.06683272 -0.4716481 -0.6018041 0.55446416 0.5218706 0.5022292 0.71278435 -0.7423482 0.65861136 -0.77672714 0.6058258 0.528827 -0.55322206 -0.6638777 0.5633302; 0.77095294 0.56496936 -0.43352574 0.5651581 0.76388097 -0.6744644 -0.7160639 0.5766815 -0.21686372 0.5630114 -0.78412765 -0.41623342 0.5908139 -0.5575324 -0.6427041 -0.64375997 0.4080602 -0.46174484 0.65291214 -0.67766136 -0.66337913 0.5156947 0.72540736 -0.39357066 0.49640006 0.42202416 0.51918346 -0.43049482 0.64494383 0.581099 -0.5340163 -0.7903437 0.52380306 -0.6260459 0.7102788 -0.45177588 0.73533493 -0.56903523 0.5477192 -0.6229005 -0.53235936 -0.5275439 0.6969577 -0.4917919 0.49601707 -0.4335438 0.6113882 0.010564684 -0.5507497 -0.4312191 -0.54784346 0.5552444 -0.60646826 -0.5281563 0.75997984 0.5386584 -0.52771854 -0.5027205 0.7469965 0.6931116 -0.6023702 -0.515809 -0.75492066 0.85804 -0.8126407 0.55336034 0.6096358 0.7872682 -0.531736 -0.7854019 -0.43790478 -0.77600676 0.61466026 -0.45771343 -0.5864831 -0.4181661 -0.67392236 0.50863886 0.39600477 -0.52193093; -0.75293326 -0.42458707 0.6434641 -0.5597291 -0.67159027 0.5502091 -0.41807112 -0.7451969 -0.81005687 -0.5111375 0.5090295 0.5904212 -0.5833519 0.6068203 0.75050235 -0.7592377 -0.4059094 0.76784986 -0.4640027 0.586649 0.6691707 -0.5651201 -0.5720127 0.26880392 -0.59610546 -0.4486724 -0.64251083 0.6320382 -0.47947356 -0.63332725 0.44763225 0.6743427 -0.44548848 0.6769823 -0.45867988 0.5975822 -0.80077744 0.57144403 -0.45931408 0.6477046 0.48628828 0.4342714 -0.55043447 0.64045566 -0.57250077 0.4893327 -0.55369073 -0.60877997 0.29899114 0.61110425 0.5561056 -0.7476542 0.5386155 0.43967426 -0.52895963 -0.6577635 0.5429619 0.4345269 -0.7629135 -0.40446538 0.48935017 0.6615625 0.45309255 -0.6975812 0.67605036 0.7928462 -0.6935737 -0.62284493 0.59945416 0.6109611 0.42182767 0.75894207 -0.42623273 -0.5516521 -0.8079817 0.7597095 0.66544837 -0.5086654 -0.40615785 0.54936916; -0.38761804 -0.5823541 0.6399032 -0.65450245 -0.5778997 0.7096617 0.6893687 -0.7461618 -0.46062616 -0.54898196 0.4314499 0.7250569 -0.37699306 0.43908605 0.6911652 0.50543094 -0.8106181 0.4603948 -0.75412863 0.57946 0.5528339 -0.64284295 -0.5089371 0.75692695 -0.674737 -0.44955778 -0.5260573 0.53544813 -0.6808836 -0.53144205 0.48261276 0.7641347 -0.64169323 0.7369333 -0.52437615 0.49000227 -0.69624895 0.7710941 -0.6761009 0.50174874 0.7928248 0.77613926 -0.72280246 0.7155647 -0.47859594 0.7124775 -0.53112566 -0.47921497 0.4588427 0.7067729 0.59646595 -0.46024385 0.8008714 0.5412849 -0.7180509 -0.42275015 0.7690279 0.49580106 -0.5752856 -0.6840186 0.7953591 0.59824556 0.76318425 -0.35267913 0.5527456 0.45160875 -0.71762705 -0.7141115 0.5732883 0.5465032 0.57064605 0.5430239 -0.65489525 -0.74691546 -0.47782263 0.73139656 0.5623322 -0.55730104 -0.6370271 0.47016338; -0.6355024 -0.37690055 0.5115154 -0.7379821 -0.7084984 0.7501694 -0.59429324 -0.5455583 -0.419837 -0.7245142 0.61079085 0.6291031 -0.6942375 0.55180174 0.69590366 1.4261591 -0.55675185 0.7019029 -0.771087 0.7190251 0.4941347 -0.6155806 -0.72278523 0.55897945 -0.5768249 -0.44716313 -0.567462 0.5711407 -0.39953387 -0.43346837 0.72395736 0.51332337 -0.45772156 0.49403727 -0.57011837 0.6004958 -0.666165 0.66056025 -0.5958499 0.5187284 0.71198535 0.6204937 -0.60993063 0.60421425 -0.7445116 0.8115103 -0.4682831 -0.62730855 0.635078 0.39881858 0.66286546 -0.7040355 0.4765382 0.4734088 -0.5187612 -0.6747745 0.82654476 0.71905744 -0.54049367 -0.49875274 0.59874713 0.6039335 0.5089801 -0.5916596 0.42659673 0.5776929 -0.52463526 -0.44732958 0.7775882 0.7267395 0.71726406 0.72707546 -0.7952843 -0.6704094 -0.69574887 0.5368924 0.75549054 -0.49020758 -0.49533865 0.76691514; -0.56670195 -0.75781 0.44036657 -0.4234727 -0.4019571 0.4818724 0.69885355 -0.5420429 0.38535017 -0.76346284 0.79371685 0.7574695 -0.60494995 0.7486702 0.700649 0.64260256 -0.573226 0.6186357 -0.7077097 0.4777807 0.76308596 -0.5144482 -0.58902985 0.4338618 -0.64333564 -0.5840853 -0.4995372 0.46063265 -0.7083126 -0.62219864 0.6480119 0.56587243 -0.49050245 0.5211589 -0.6422859 0.4747945 -0.52453077 0.66361356 -0.43311355 0.6010684 0.4422257 0.72517204 -0.6322426 0.5242902 -0.7461021 0.6130077 -0.8028582 0.7364943 0.7366446 0.7355919 0.5371828 -0.6506191 0.42500165 0.45477614 -0.7590935 -0.7805481 0.41602704 0.59319764 -0.4785006 -0.39644587 0.40217873 0.5087749 0.42466572 -0.7369289 0.11216948 -0.6434861 -0.5819357 -0.52635455 0.6250788 0.64817935 0.4982418 0.6531149 -0.48095453 0.52938074 0.6252887 0.47523886 0.7834588 -0.46611324 -0.08831305 0.78006357; -0.4941368 -0.7351676 0.5991359 -0.61220074 -0.5950303 0.67666966 -0.6572168 -0.7736886 -0.6040559 -0.5411309 0.6176424 0.78620124 -0.677314 0.60666335 0.5659879 -0.59208316 -0.7365422 0.591852 -0.6956716 0.69448423 0.6852443 -0.45413312 -0.7257934 -0.45623776 -0.6507939 -0.7303532 -0.72758996 0.6273891 -0.7120763 -0.6242589 0.6678456 0.045537572 -0.7395731 0.48955134 -0.58569044 0.1917354 -0.48043177 0.3619886 -0.5662771 0.7645129 0.4691185 0.6561718 -0.4572388 0.64679706 -0.7332932 0.44957608 -0.73427576 -0.51533455 -0.6212105 0.7007251 -0.5603461 -0.7778814 0.4626853 0.56464636 -0.72380584 -0.55447644 0.34616163 0.64589 0.29696777 -0.66489077 0.48463503 0.6620859 0.6985822 -0.6343205 0.730508 0.54475754 -0.79168427 -0.5684807 0.5924353 0.5142779 0.59744734 0.55359715 -0.64997435 -0.5647386 -0.5519234 0.6496309 0.7568169 -0.64196 -0.42873818 0.6048574; 0.39839554 0.34577847 0.76775444 -0.757817 -0.6696201 0.6197452 -0.5096222 0.49402845 -0.6988892 -0.82344294 0.549176 -0.45474023 0.56980747 0.64844793 0.4211709 -0.44557923 -0.5193981 0.5404088 0.30624458 0.7372454 0.63459605 0.50733036 -0.7257353 -0.74183553 -0.012271443 -0.5665615 0.4170928 -0.69211084 -0.6941845 0.51890355 -0.6473918 -0.583846 0.7313145 0.63220865 -0.60986876 -0.5721122 -0.43352732 -0.4516786 -0.51062155 0.12739232 0.6317589 0.81897265 -0.6347932 0.6612735 -0.7528437 0.030262897 0.6930952 -0.6495488 -0.5427131 -0.3925517 -0.65789455 -0.6393269 0.43871498 0.81144327 0.3743865 -0.5083931 -0.41331482 -0.65307873 0.6774027 -0.65291333 0.4432896 0.63070816 -0.1252651 -0.5167733 0.16419911 0.725672 -0.5738973 -0.4901023 -0.48099 -0.24865367 0.5375447 0.44965783 0.5326327 -0.5181072 -0.49386042 0.37849852 0.73124164 -0.7462316 -0.7188915 0.69033664; -0.61791193 -0.6388077 0.48562515 -0.77659297 -0.8098663 0.5075401 0.6486279 -0.71869195 -0.4462635 -0.75042295 0.49526718 0.6330362 -0.6864645 0.6717266 0.5922414 0.8392434 -0.6808459 0.45202577 -0.64881605 0.70461303 0.7347203 -0.62922823 -0.6309448 0.49301592 -0.7730006 -0.4843366 -0.5195573 0.5112789 -0.7638222 -0.5000404 0.81420624 0.6456495 -0.39593047 0.49106 -0.49317926 0.47577065 -0.37946427 0.57433164 -0.7203389 0.7261122 0.6391729 0.5089215 -0.60651433 0.49003467 -0.5345399 0.664571 -0.60890794 -0.58876604 0.8521746 0.61613244 0.4030618 -0.78001183 0.49539703 0.63184065 -0.65776896 -0.736898 0.5116109 0.7753953 -0.41459608 -0.71591747 0.54438937 0.6140851 0.5832294 -0.5563877 0.16759785 0.5420737 -0.6713155 -0.46287543 0.78050196 0.48685005 0.48481 0.59118044 -0.56529564 -0.7053723 -0.3354875 0.7986553 0.62318283 -0.5019917 -0.76354706 0.5349132; -0.7598965 -0.60971034 0.43547246 -0.6552839 -0.5853681 0.5881628 0.7799085 -0.6038985 -0.699894 -0.48549426 0.42392698 0.44699404 -0.6392756 0.5956336 0.45961595 0.6376641 -0.6999867 0.6412813 -0.6077857 0.5262903 0.69722587 -0.43904904 -0.42783237 0.506183 -0.66903114 -0.55725944 -0.7537053 0.6370396 -0.5888138 -0.7465834 0.53613806 0.7198797 -0.7538478 0.503 -0.4725647 0.44182184 -0.4230596 0.47678015 -0.5608107 0.517282 0.4770298 0.72814536 -0.6958922 0.53379524 -0.40386322 0.44499585 -0.66032743 -0.44346198 0.6841031 0.5122591 0.7080598 -0.59780484 0.5669289 0.71167773 -0.5763658 -0.78647214 0.40116346 0.6762798 -0.6314219 -0.5233226 0.55489045 0.7256514 0.6233128 -0.6910102 0.6916285 0.5047528 -0.74191 -0.65116817 0.76896894 0.6471132 0.40822664 0.72637963 -0.5043218 0.5129211 -0.7681224 0.3995295 0.6352857 -0.40256378 -0.47744414 0.62874216; 0.65263784 0.6371609 -0.6798184 0.5472701 0.6373426 -0.8470342 -0.56838983 0.46391714 -0.16815145 0.73296505 -0.58310735 -0.8191319 0.8382224 -0.5145446 -0.7783344 -0.53210175 0.45763373 -0.7509989 0.7448322 -0.68335867 -0.6607319 0.5033618 0.5255498 -0.52002156 0.5915435 0.4456002 0.609772 -0.4328835 0.51227707 0.568279 -0.40461737 -0.47128716 0.525294 -0.6352121 0.5060226 -0.7092602 0.8219172 -0.4681639 0.746677 -0.44266424 -0.69233984 -0.77275056 0.4614869 -0.43947685 0.6543892 -0.522444 0.83857226 0.10345073 -0.5203028 -0.59283835 -0.7988136 0.37386993 -0.43753552 -0.7223294 0.806853 0.7508437 -0.30459878 -0.68904644 0.54435253 0.7313022 -0.40301213 -0.76714545 -0.61362696 0.6156989 -0.5808832 0.5829027 0.8113018 0.61796904 -0.40878612 -0.5582606 -0.5212843 -0.47900793 0.562135 -0.64629185 -0.3476614 -0.44113097 -0.78767335 0.6576588 0.10312837 -0.6089297; -0.5597414 -0.69148314 0.48367688 -0.68716407 -0.5796063 0.45386446 -1.0273129 -0.46935833 -0.58020407 -0.48524013 0.68952155 0.46308783 -0.6913587 0.4421274 0.76174384 0.4753743 -0.6088065 0.5248196 -0.45882705 0.4527257 0.6383424 -0.62623286 -0.4646669 0.7284229 -0.5638598 -0.51899385 -0.61184627 0.7500547 -0.66203606 -0.6626217 0.6535004 0.728907 -0.69570047 0.40821642 -0.61418587 0.5272572 -0.47925404 0.5349082 -0.669927 0.43041041 0.75780606 0.4746483 -0.47159988 0.40934554 -0.6913764 0.4648329 -0.55203307 -0.7346966 0.618657 0.45334873 0.70319253 -0.69655323 0.40241626 0.56810856 -0.5138387 -0.574997 0.71750766 0.7896438 -0.59538674 -0.51683676 0.59184575 0.6389045 0.5814811 -0.4999842 0.71011055 0.6969742 -0.74203044 -0.7685445 0.66740495 0.5213579 0.691753 0.76634467 -0.51589036 -0.61799574 -0.74756885 0.5901036 0.6582453 -0.52461046 -0.47887552 0.6533246; 0.41639876 0.64621407 -0.54880625 0.59622526 0.48585364 -0.50304264 0.6705488 0.7320623 0.48024672 0.58439404 -0.40769863 -0.41619036 0.7062401 -0.63208383 -0.6160282 0.541546 0.67061156 -0.7152876 0.39369872 -0.5960474 -0.39954767 0.690793 0.7859613 -0.7152288 0.5665998 0.58853257 0.56229085 -0.7796047 0.5557904 0.6567051 -0.7601148 -0.7112968 0.81625545 -0.47192463 0.61932796 -0.3776854 0.7578546 -0.5276036 0.73658925 -0.4209271 -0.51347005 -0.6060306 0.76231277 -0.4804839 0.61966336 -0.76876926 0.5158962 0.5803257 -0.43633512 -0.43721846 -0.5059741 0.49468008 -0.4187276 -0.5500116 0.5817209 0.38632795 -0.737143 -0.72511846 0.8128404 0.71274704 -0.67058265 -0.63619256 -0.6124323 0.41728336 -0.7400552 -0.5026582 0.42743173 0.71930915 -0.45138317 -0.5913145 -0.42437166 -0.53051955 0.5059199 0.73039854 0.57138795 -0.5203383 -0.4709891 0.51470345 0.5084189 -0.4035353; 0.81468177 0.52880174 -0.7596556 0.5834108 0.74337304 -0.43262246 -0.6308846 0.6949509 0.63560504 0.78583395 -0.4913036 -0.61079365 0.7528584 -0.42385656 -0.66173345 -0.7164861 0.56543225 -0.52021945 0.733751 -0.5397116 -0.69358605 0.6746681 0.54558766 -0.44647527 0.77680266 0.77382475 0.43425423 -0.4504719 0.6686998 0.5610319 -0.5814027 -0.7426727 0.7119159 -0.6735461 0.4121175 -0.5362901 0.7585764 -0.51174134 0.59007096 -0.66342425 -0.60092825 -0.58479273 0.6286308 -0.57799286 0.5213692 -0.43697634 0.6610711 0.6635667 -0.5558996 -0.66272295 -0.5617338 0.5283241 -0.7715207 -0.70642334 0.64440686 0.5285156 -0.43531597 -0.7696546 0.45061377 0.6335656 -0.43140897 -0.67113423 -0.71089643 0.4820623 -0.62170726 -0.72255194 0.68630546 0.73285335 -0.43446139 -0.4704908 -0.59211665 -0.55822015 0.44628158 0.33167058 0.51344645 -0.52053225 -0.46069998 0.5175805 0.6448598 -0.63545483; -0.7612836 -0.45840374 0.53389144 -0.6790171 -0.6873523 -0.7300036 0.564837 -0.6891884 0.7572011 -0.4175978 -0.57494956 0.50269586 -0.56521034 -0.8026953 -0.6757382 0.44143513 -0.65990186 0.37285194 -0.42846724 -0.52660745 0.6291088 -0.45372522 0.46818826 0.7575894 -0.6968873 -0.6724295 -0.5535869 0.77337927 -0.40157613 -0.552668 0.72914517 0.6089898 -0.659338 0.4186669 -0.5820204 0.40081784 0.41584706 0.7442564 0.81746227 0.47377637 0.464405 0.4421776 -0.47104535 0.28350735 0.6329794 0.59279877 -0.52923197 0.60974115 0.71876264 0.5216431 0.5550561 0.7059143 0.7231446 -0.4527537 -0.5571262 -0.49817073 0.6450219 0.6008968 -0.6323259 0.7602045 -0.6077469 0.7398965 0.48258018 0.4575255 -0.6047038 -0.5959763 -0.4992888 0.38403746 0.60137504 0.65245014 0.62419665 0.65539324 -0.66570455 0.6943609 0.4291537 0.50425243 0.6953414 -0.35039812 0.58401024 0.44851238; -0.7804974 -0.743775 0.6833573 -0.70970565 -0.55601776 0.6836369 0.60380733 -0.7002346 0.27865997 -0.6092258 0.6744001 0.80537397 -0.6289038 0.5300059 0.6161682 0.5034044 -0.44996798 0.55253136 -0.6417515 0.41861105 0.5703038 -0.6027628 -0.5752719 0.5287103 -0.56356514 -0.47411725 -0.63830316 0.5640146 -0.73720473 -0.6194337 0.58095115 0.57612854 -0.46386075 0.6624928 -0.74239415 0.48356342 -0.76568836 0.41045564 -0.39342237 0.7547251 0.5719708 0.48166272 -0.4973281 0.70619184 -0.4458008 0.49129257 -0.516559 1.1644361 0.6005152 0.5256599 0.59945035 -0.4976396 0.59458625 0.5113978 -0.6087414 -0.5057175 0.42437595 0.72486913 -0.49210626 -0.5384799 0.51395994 0.7041249 0.5378509 -0.65363044 0.42470416 -0.5207154 -0.53598386 -0.84141207 0.6347313 0.61394894 0.7407315 0.58738077 -0.6582184 0.8550814 0.3951655 0.43500012 0.57950306 -0.6748016 0.08280776 0.70192224; 0.498269 0.5308654 -0.59215903 0.5109282 0.5869529 -0.41321233 0.8764939 0.7428965 0.36591235 0.6678432 -0.3915749 -0.5082603 0.38207453 -0.5396662 -0.44062236 0.33852062 0.63371927 -0.5713505 0.44459236 -0.5284988 -0.436472 0.6873236 0.7721154 -0.84618324 0.4690652 0.4372323 0.585722 -0.711139 0.52108485 0.69354343 -0.76619095 -0.5173946 0.76752156 -0.77544826 0.5270338 -0.7089231 0.5941317 -0.691062 0.5004246 -0.63969964 -0.60164195 -0.7419398 0.46990857 -0.658371 0.6874613 -0.7833305 0.52663887 0.61907744 -0.51778793 -0.57346946 -0.36733913 0.41859326 -0.51546246 -0.75021523 0.7358853 0.66979176 -0.72429466 -0.71126294 0.6663045 0.46915478 -0.5093785 -0.4819295 -0.76329654 0.55559736 -0.65373135 -0.7397395 0.4077085 0.56947225 -0.71903616 -0.6632504 -0.71093327 -0.49129444 0.46371606 0.63701916 0.71816796 -0.4867562 -0.6562848 0.42507458 0.6740598 -0.4383902; 0.4492641 0.6783613 -0.62559366 0.7226752 0.5428842 -0.5709028 -0.74253446 0.41221857 0.50386775 0.6360169 -0.72741693 -0.6286566 0.62379164 -0.43527326 -0.46527794 -0.38401973 0.4035512 -0.4941514 0.7396433 -0.66995984 -0.72603583 0.53132826 0.4794788 -0.40455234 0.6350327 0.4696713 0.48209226 -0.6027119 0.7231431 0.43667397 -0.51763797 -0.7339112 0.8027371 -0.5625465 0.48287463 -0.6368775 0.7715174 -0.56066364 0.49051595 -0.7379955 -0.54010636 -0.64250386 0.74900514 -0.42427975 0.47350198 -0.50326437 0.64008343 0.5087637 -0.6979958 -0.71396196 -0.504739 0.44950184 -0.7183755 -0.58145666 0.4434357 0.65376335 -0.5242622 -0.5666606 0.55298823 0.40657204 -0.50027514 -0.532671 -0.5721096 0.7626391 -0.6110381 -0.58347386 0.5508722 0.77521807 -0.67765 -0.42991212 -0.52311575 -0.65314084 0.49055356 -1.207291 0.77760994 -0.5729774 -0.598502 0.5113627 0.62195545 -0.6453084; 0.48295453 0.67109245 -0.75386506 0.56409913 0.6285574 -0.6900574 0.7379818 0.6698604 0.4921617 0.64733714 -0.74626756 -0.5907402 0.62457496 -0.7272471 -0.7247572 -0.44257846 0.56520295 -0.5503764 0.47525024 -0.76768804 -0.6400966 0.62612647 0.641807 -0.5374069 0.4196178 0.47334343 0.6754808 -0.48221186 0.67636955 0.5154806 -0.7609005 -0.58698153 0.45165533 -0.43627048 0.49643013 -0.5307486 0.70242816 -0.4412696 0.53529125 -0.5709532 -0.64721376 -0.42723593 0.52734756 -0.7431725 0.63473105 -0.71549267 0.49021655 0.44037932 -0.7024335 -0.7014024 -0.41939276 0.43552724 -0.5504013 -0.46240392 0.69373286 0.61941755 -0.55088794 -0.60335034 0.5072065 0.5828243 -0.67153883 -0.5950318 -0.68336296 0.7703679 -0.6752528 -0.4411028 0.67945457 0.6856605 -0.770626 -0.43977183 -0.45672128 -0.67599505 0.49037227 0.51399016 0.6196205 -0.52616644 -0.3988312 0.43543878 0.5416295 -0.7369772; 0.5608405 0.5522952 -0.59388536 0.6258609 0.40153837 -0.8354255 -0.5412381 0.7084847 -0.03694522 0.5889959 -0.8088794 -0.7710419 0.53461945 -0.74647 -0.5105228 -0.68035454 0.49968064 -0.595556 0.65363 -0.68245435 -0.63315564 0.53179944 0.4783312 -0.6959573 0.7681054 0.5961255 0.7419014 -0.68994606 0.6434498 0.714695 -0.46568173 -0.6452619 0.60460705 -0.6401227 0.59617126 -0.5155125 0.6144863 -0.6633225 0.37856036 -0.5328653 -0.43481207 -0.6648146 0.53660196 -0.6426882 0.3943762 -0.44864267 0.5248145 0.55165166 -0.5712701 -0.68633336 -0.475862 0.4038676 -0.7350573 -0.7054918 0.6451908 0.7762783 -0.3418658 -0.6286112 0.5159598 0.5835165 -0.4258637 -0.60608953 -0.42362532 0.83338004 -0.76087266 0.48167798 0.6054046 0.8220713 -0.49199024 -0.7134229 -0.4184832 -0.45182645 0.5978514 -0.8941627 0.20205669 -0.55484575 -0.7477905 0.59773946 0.62574524 -0.7383457; -0.67933154 -0.52865565 0.7731535 -0.5988476 -0.5206717 -0.5613036 0.77095777 -0.52403873 0.52461547 -0.6656456 0.4416039 0.48035425 -0.61879253 -0.5387678 0.56960136 0.63753116 -0.6640966 0.722826 -0.54439884 -0.3069624 0.70764613 -0.517877 0.679689 0.70188814 -0.43551353 -0.6039736 -0.5987606 0.6742021 -0.39942694 -0.47570542 0.75543165 0.567328 -0.5214799 0.6254402 -0.7916039 0.52306116 0.948036 0.6654144 0.2246675 0.5230565 0.5142386 0.5961478 -0.5481535 0.74137634 -0.52439535 0.57907176 -0.47798854 0.40278518 0.73502827 0.4758885 0.73044354 -0.93596816 0.4723394 -0.5027092 -0.56929624 -0.42044327 0.74070853 0.57848436 -0.41508365 -0.5425651 -0.40219647 0.7024915 0.7340604 0.5656569 -0.7204927 -0.46881694 -0.6627924 -0.08233627 0.6102106 0.6560339 0.5900668 0.6372418 -0.5696555 0.472212 0.49101996 0.5358673 0.7415994 -0.67110497 0.65849745 0.6720025; -0.61704504 -0.6693309 0.7005566 -0.7062809 -0.70274884 0.657857 0.55695766 -0.6982327 0.49679834 -0.6190519 0.53934467 0.5836353 -0.5959669 0.52635616 0.49258828 0.7092848 -0.5442244 0.43183675 -0.6107885 0.5944031 0.562677 -0.58444065 -0.6659154 0.61184776 -0.743735 -0.5978175 -0.69996303 0.69907176 -0.58895355 -0.52404594 0.435733 0.65818703 -0.5407716 0.59486294 -0.74956846 0.5523113 -0.7153387 0.53304696 -0.4788631 0.5797563 0.57713115 0.50249195 -0.6861376 0.6020809 -0.5009486 0.7828327 -0.5394928 -0.6054638 0.5170456 0.4807327 0.72765225 -0.57064605 0.6818567 0.59242654 -0.52804905 -0.59063613 0.544256 0.38918436 -0.689512 -0.7637492 0.57412624 0.66051704 0.53555804 -0.75749785 0.60448253 -0.63513863 -0.7272207 -0.7070383 0.67986345 0.606311 0.5843338 0.54419065 -0.75915426 0.53288406 0.27581704 0.7755805 0.541213 -0.5107047 -0.6817705 0.7442534; 0.36461294 0.6897924 -0.7283588 0.7515633 0.57352746 -0.7528699 0.45757598 -0.32165003 0.78147364 0.830731 -0.46182373 -0.72500545 0.60603184 -0.6359838 -0.7136374 0.6165739 0.548113 -0.72424704 0.69407415 -0.5031445 -0.68321836 0.34258136 0.7133789 0.57800514 0.49627772 0.41161788 0.7523431 0.68634754 0.8155263 0.4001696 0.09642009 0.7493398 0.757121 -0.54927015 0.71542686 0.54433423 0.6511527 0.48255053 0.75410384 -0.3864086 -0.74245167 -0.5640644 0.6026096 -0.45464867 0.74712497 -0.56772286 -0.38469243 0.55528307 0.7281372 0.35871834 0.6431263 0.48894304 -0.51797277 -0.5723251 -0.36955598 0.8126369 -0.4328804 -0.2725629 -0.40641394 0.73579645 -0.5719844 -0.6350507 -0.59987456 0.57444674 -0.4774795 -0.5688367 0.5718143 0.5259624 -0.21726641 -0.701249 -0.7888852 -0.66558546 0.40935862 0.43072644 0.7332962 -0.5543382 -0.711069 0.4479385 0.6040846 -0.5161542; -0.80614793 -0.8090481 0.4191235 -0.6223618 -0.7715732 0.4815169 -0.47905174 -0.73330945 -0.78234005 -0.6352236 0.7309707 0.7369925 -0.4656511 0.47483626 0.5334619 -0.69836897 -0.7008479 0.4986487 -0.7364563 0.5389051 0.59162205 -0.5147553 -0.62784946 0.41842005 -0.72324157 -0.77568835 -0.6898115 0.4282222 -0.6801496 -0.40567142 0.5175262 0.4974164 -0.74302113 0.77973807 -0.66056025 0.62046117 -0.81575286 0.45762408 -0.60757464 0.6023734 0.72885156 0.5887513 -0.61135226 0.7053981 -0.48620865 0.62368387 -0.7431944 -0.68825936 0.44831505 0.7817598 0.60766655 -0.598755 0.7556331 0.67267466 -0.6362627 -0.61167425 0.56024903 0.7608011 -0.36899543 -0.47621602 0.594014 0.5945635 0.64091504 -0.81820655 0.43830493 0.67675173 -0.6285563 -0.45351216 0.69011647 0.45631924 0.77262336 0.5403421 -0.76933914 -0.5688093 -0.6513797 0.46789646 0.7014554 -0.7290549 -0.38919738 0.74392426], bias = Float32[-0.60930264; -0.6048689; -0.059062093; -0.68409634; -0.60262835; 0.6197492; -0.6146431; -0.66844296; 0.59880495; 0.59734607; -0.6291986; -0.60940903; 0.59203994; 0.5969653; -0.5957693; 0.6193033; 0.6354846; 0.6094279; 0.6856425; 0.56867284; 0.6173876; 0.5322377; -0.5331827; -0.60413396; -0.5880598; 0.50907946; 0.5902003; 0.5878925; 0.60322183; 0.5668153; -0.60678613; 0.6296705; 0.60037154; -0.566149; -0.5367215; 0.5993316; -0.5958223; 0.5554202; 0.5918224; -0.60278696; -0.6175806; 0.6083491; 0.6041506; 0.5901036; -0.60786974; 0.48701137; -0.55870897; -0.5672204; 0.60962886; 0.6443544; -0.6244659; -0.5613323; -0.6082326; -0.56592387; 0.61496824; -0.6027066; 0.61434215; -0.63329417; 0.6117565; 0.564371; 0.5642497; 0.6420011; 0.577948; 0.51172715; 0.55324507; 0.60293293; -0.6205885; 0.6013484; -0.5901202; -0.60347164; -0.4543302; 0.6246362; -0.59404975; -0.60573345; -0.58755994; -0.62447035; 0.15845023; 0.5726274; -0.5629181; 0.5810352]), layer_3 = (weight = Float32[-0.59612274 0.7105092 -0.5085276 -0.5819301 -0.44578862 0.56354874 -0.6052022 -0.7194084 0.52046454 0.47439224 -0.6591354 -0.5310527 0.34268618 -0.5990739 -0.4963576 0.5266145 0.65011895 0.37216917 -0.46920502 0.3080809 0.36851466 0.33560207 0.8882648 -0.4228902 -0.40172756 0.51375616 0.33433273 -0.653005 0.42733708 0.46983245 -0.49212432 -0.79448634 -0.62225956 0.7936645 0.8274597 0.7114333 0.7152472 -0.7685745 0.40227672 -0.35751292 -0.5975502 0.51098293 0.37659195 0.35144943 -0.48341575 -0.68506104 0.7358365 0.7228138 -0.52920413 -0.7223784 0.6377098 0.77293926 -0.5148605 0.7435852 0.6331293 -0.3472004 -0.55676854 -0.67175925 0.40688547 -0.7532966 -0.8286071 -0.4683019 -0.76559496 -0.7525486 -0.7219953 0.41079786 0.5162892 0.4154412 -0.6248857 0.68134755 0.6952521 -0.5178153 -0.4741478 -0.4966736 -0.48459896 0.5178893 -0.3148496 -0.53865683 0.7733024 -0.64413923; -0.36411256 0.7532383 -0.9895923 -0.5176107 -0.66511697 0.5042151 -0.36780247 -0.34362423 0.51208997 0.55855733 -0.3280677 -0.39849165 0.5890005 -0.76474714 -0.38261932 0.17153278 0.33394006 0.5641767 -0.83308524 0.7763703 0.66599476 -0.90047306 0.29113743 -0.4454386 -0.590075 0.6679651 0.61633176 -0.6420139 0.6722711 0.70279557 -0.64470404 0.6955295 -0.73099965 0.5189792 0.38742232 0.40379435 0.6588473 -0.8548986 0.60406554 -0.33913937 -0.42062256 0.3321047 0.36439666 0.6344727 -0.58291876 -0.5267411 0.79401886 0.5532648 -0.7457174 0.27417213 -0.3616792 0.5663152 -0.4153524 0.7864794 0.41232586 -0.32839924 -0.8822944 -0.3843342 0.5830688 -0.9009973 -0.59835494 -1.0863532 -0.5339779 -0.44966733 -0.78697324 0.3889594 0.8167737 0.53096277 -0.5176402 0.8260096 0.44280365 -0.8561222 -0.601831 -0.32554758 -0.44351822 0.8550939 0.20963205 -0.9128947 0.41619325 -0.7001883], bias = Float32[-0.5726958; -0.598807]))

loss_neuralode(p_trained_moderatenoise)


#High noise loss collection:
#---------------------------
#-----------------

rng = Random.default_rng()
Random.seed!(99)

x1=Array(sol)

x1_mean = mean(x1, dims = 2)
noise_magnitude = 35e-2
x1_noise = x1 .+ (noise_magnitude*x1) .* randn(eltype(x1), size(x1))
#Displaying true data vs noisy data
dudt2 = Lux.Chain(Lux.Dense(2, 80, tanh),Lux.Dense(80, 80, tanh), Lux.Dense(80, 2))
#Setting up the NN parameters randomly using the rng instance
p, st = Lux.setup(rng, dudt2)


true_data= x1_noise[:,1:end-10]
### Define loss function as the difference between actual ground truth data and Neural ODE prediction
function loss_neuralode(p)
    pred = predict_neuralode(p)
    loss = sum(abs2, true_data .- pred)
    return loss, pred
end
p_trained_highnoise = (layer_1 = (weight = Float32[0.6558051 -1.9769568; -0.96869016 -0.8651783; -0.35831967 -0.23670292; 0.35312834 0.37676066; 0.33044958 0.38961545; 1.1257575 0.61379933; 1.1622341 0.596189; 0.4028757 -1.4861126; -0.61400497 -0.6700542; 0.0291736 -1.5930948; 0.93559164 0.69817203; 0.121828 1.3583; -0.65842396 -0.7604909; -0.4461029 -0.5282506; 0.9981012 0.71944237; -0.3970528 -0.644005; 0.29753482 0.16408429; -0.31872532 -0.27865088; -0.34146693 -1.1026759; -0.35264865 -0.5321799; -0.6076075 1.0302732; -0.51396096 -0.92450047; -1.1459836 0.46048316; -0.23684168 -0.53882927; -0.6403474 -1.0526239; 0.012370507 0.10881664; -0.6100859 -1.1797917; -0.31876594 -0.20889916; -1.008588 -0.8960992; -0.075161174 0.20624746; -0.383927 -0.44160447; -0.95197266 -0.46020168; -0.88461936 -0.24794246; -0.32351616 -0.18787435; 0.36860788 0.39631006; -0.33532023 -0.47161844; -0.94404894 -0.6368902; -0.36127472 -0.3774073; 0.34813702 0.51051044; -0.375305 -0.22407101; -0.3254487 -0.2264224; -0.33765522 -0.39119425; -0.8334225 -0.9024241; -0.39780828 -0.49529716; 0.3872502 0.48270953; -0.35107526 -0.45145255; -0.5172893 -0.82782847; 0.41902134 0.49432766; -0.29859602 -0.5446888; -0.13681418 1.6891565; 0.67620975 0.7941255; 0.39512804 0.48452246; -0.30001977 -0.47048098; -0.32144877 -0.5749233; -0.6632864 -0.82315177; -0.9836275 -0.91904634; -0.44698137 -0.5058364; 0.39975637 1.0189162; 0.026436368 0.3139115; 0.29783627 0.5250002; -0.3607462 -0.57095766; -0.29365453 -0.19132349; -0.28479064 -0.27854812; -1.1756406 -0.62353945; 1.1110522 -0.5326582; 0.6249973 0.623554; -0.8784806 -0.7153809; -0.96295464 -0.66760594; -0.3992144 -0.47764754; 0.572013 0.9504402; -0.35035452 -0.34676108; 0.8050739 0.8770821; 0.34821886 0.17728212; 0.9250332 0.3968947; -0.7577291 -0.6174782; -0.38194153 -0.46657848; 1.1694255 0.8339904; 0.32054856 0.325712; 0.44144517 0.5438747; 0.6327066 0.9666276], bias = Float32[-0.21293104; -0.509192; -0.58420056; 0.6490055; 0.5906057; 0.63916016; 0.7199837; -0.2182945; -0.6219794; 0.015808446; 0.612537; 0.15952247; -0.6113903; -0.55565256; 0.57376504; -0.5124538; 0.9073314; -0.6158466; -0.54247653; -0.4449271; -0.6355151; -0.6452891; -0.54687047; -0.21604553; -0.34011155; 0.18164015; -0.29651725; -0.2296963; -0.6177089; -0.17639033; -0.4624117; 0.60107124; -0.7398981; -0.59084386; 0.6456671; -0.4488908; -0.7104505; -0.4132137; 0.4403872; -0.6050884; -0.604525; -0.563872; -0.6561983; -0.63359404; 0.6678978; -0.6184023; -0.6257269; 0.35435444; -0.44801837; 0.4434472; 0.87493294; 0.56319875; -0.514512; -0.41821373; -0.66511613; -0.6286689; -0.6370794; 0.6052622; -0.7408728; 0.4626727; -0.42989612; -0.73268837; -0.27104372; -0.6204835; 0.49357253; 0.74883765; -0.65877557; -0.757757; -0.5755981; 0.5155954; -0.606771; 0.41979107; 0.4707718; 0.8670253; -0.8109737; -0.63032365; 0.5641233; 0.54093623; 0.5594434; 0.5724111]), layer_2 = (weight = Float32[0.595844 0.62255746 -0.6048064 0.47088915 0.56894326 -0.3714719 -0.7494048 0.49809265 -0.5886921 0.6527498 -0.60115796 -0.57274354 0.4710216 -0.55110615 -0.49480703 -0.5130665 0.5020461 -0.4851048 0.5388582 -0.61023194 -0.42084867 0.48529467 0.46111476 -0.49684012 0.38586795 0.5615127 0.47382137 -0.7101357 0.5744599 0.5564112 -0.4852072 -0.51571614 0.61966676 -0.58179355 0.52461493 -0.57395065 0.47362188 -0.74685156 0.5371282 -0.7495093 -0.66394454 -0.5715553 0.43539843 -0.4408686 0.467619 -0.7662881 0.5328277 0.6313273 -0.5657455 -0.721705 -0.7794773 0.72068375 -0.57117766 -0.64977443 0.69758904 0.6003334 -0.52000266 -0.6805306 0.5645022 0.42378187 -0.72917175 -0.6944366 -0.79525703 0.39885733 -0.39507154 0.6114697 0.6143689 0.4426371 -0.55741286 -0.41204518 -0.64024705 -0.7546499 0.79502195 -0.6766514 -0.33535653 -0.7425691 -0.7025645 0.54622054 0.6606563 -0.49524286; 0.7791602 0.46131268 -0.68960685 0.56233764 0.4581871 -0.4724067 0.051426936 0.6146325 0.4658337 0.48359835 -0.6693946 -0.50086576 0.66221684 -0.5738178 -0.48955294 -0.15024637 0.7361811 -0.60817635 0.5376962 -0.7567163 -0.3853599 0.68302715 0.6477824 -0.62158775 0.31866962 0.34301814 0.49524128 -0.7245579 0.6823035 0.48666137 -0.6428638 -0.74415016 0.6365714 -0.62292117 0.60181767 -0.69837314 0.74644816 -0.675946 0.56223917 -0.7374809 -0.53175396 -0.449117 0.4045134 -0.46704087 0.57918227 -0.7993604 0.6557057 0.6990792 -0.5885078 -0.68222123 -0.8161883 0.5718204 -0.45484763 -0.57670635 0.544124 0.62360466 -0.64761484 -0.7335081 0.4603978 0.6748243 -0.7363944 -0.683331 -0.8091048 0.70289254 -0.74077106 -0.7335021 0.73255247 0.72107226 -0.42551342 -0.7464396 -0.71019626 -0.501755 0.69221556 0.35596862 0.7037271 -0.60752857 -0.49606493 0.67617196 0.4354295 -0.6652765; -0.5236022 -0.47081804 0.7514133 -0.707686 -0.6580501 -0.7073353 0.68848926 -0.53680146 0.5829231 -0.44221544 0.5529409 0.58460736 -0.5846712 -0.67308336 0.26557505 0.66919917 -0.7953678 0.53551644 -0.53407264 -0.70457494 0.532853 -0.5700839 0.83895856 0.77652335 -0.64797246 -0.5303064 -0.6651292 0.76096034 -0.683443 -0.47985375 0.54018867 0.4822639 -0.30772212 0.55062455 -0.8751148 0.52985764 0.5946266 0.5473503 0.5436129 0.6977445 0.8609611 0.77321124 -0.44018772 0.6021175 -0.6508336 0.5701013 -0.6492172 0.6193299 0.5507153 0.38455823 0.4625706 0.49487022 0.8019421 -0.6967392 -0.68525195 -0.5618426 0.7711111 0.5668334 -0.42415068 -0.707548 -0.41506997 0.82329077 0.79252917 0.55563223 -0.9951678 -0.7088832 -0.5491813 -0.313593 0.59613526 0.4198617 0.7982656 0.3394849 -0.53758705 0.5354146 0.5716574 0.8135597 0.6321579 -0.7827772 0.5843081 0.6909581; 0.5316507 0.42891204 -0.40422 0.73134035 0.56926376 -0.26650319 -0.75016654 0.4008933 -0.80595076 0.511608 -0.59143364 -0.511981 0.5947155 -0.56322616 -0.48869005 -0.6160016 0.70036244 -0.58393085 0.5451321 -0.44794086 -0.6616372 0.63628083 -0.5736109 -0.7138315 0.40849876 0.5211209 0.68778783 -0.44254494 0.5889623 0.76322186 -0.52181196 -0.744721 0.7102987 -0.78142816 0.45242092 -0.7212539 0.46720836 -0.5876985 0.5284125 -0.7029746 -0.6733066 -0.51605034 0.40757757 -0.4534494 0.57682097 -0.46153125 0.7043485 -0.2808017 -0.52702 -0.6422252 -0.450124 0.5293559 -0.61394405 -0.7038331 0.6145826 0.6479378 -0.478131 -0.56039244 0.46460497 0.63419473 0.23983128 -0.4314851 -0.4805765 -0.37709215 0.07839221 0.7043682 0.42768547 0.21004973 -0.5612638 -0.73198 -0.546911 -0.43932453 0.49866223 -0.57628286 -0.6526693 -0.7012427 -0.35799116 0.49748993 -0.32102275 -0.6100551; 0.8018856 0.84296644 -0.6560984 0.44163266 0.47735277 -0.643723 0.48070917 0.76310134 0.6917992 0.6247093 -0.59557563 -0.7904054 0.89122534 -0.575263 -0.7088146 0.6378777 0.54223734 -0.4697703 0.48151782 -0.62150246 -0.39153346 0.7599744 0.72994196 1.0515267 0.66695076 0.7254517 0.7184853 -0.42719 0.7798944 0.56159705 -0.53126675 0.8949396 0.8175477 -0.5360576 0.44205254 1.3185765 0.6206199 0.9937752 0.3942269 -0.51543295 -0.5804944 -0.51321614 0.46393415 -0.58527637 0.6306414 -0.66372424 1.0675892 0.61550426 0.91971225 -0.6520338 0.7315465 0.39210457 -0.43455595 -0.5109999 0.63005275 0.79912734 -0.14820671 -0.4543474 0.25679713 0.50841707 -0.7231702 -0.42790973 -0.5738823 0.75869006 -1.0057303 -0.69763666 0.81150156 0.5103102 -0.2909235 -0.70914066 -0.5328109 -0.6031129 0.6201941 0.7016105 0.73400146 -0.43655488 -0.6905373 0.55994874 0.5640704 -0.44317335; -0.3998228 -0.64080876 0.43935207 -0.49604896 -0.7273455 0.6906899 0.7553204 -0.5494719 0.6227164 -0.5111293 0.6493449 0.70310175 -0.5934305 0.5410186 0.6248985 0.707959 -0.54829144 0.6062491 -0.39182383 0.42632967 0.76192033 -0.62175566 -0.69173217 0.69419855 -0.56985956 -0.7811354 -0.7308386 0.4293562 -0.3985224 -0.44391164 0.5314086 0.67624545 -0.535165 0.72796696 -0.7837699 0.76090246 -0.7100478 0.45335546 -0.663019 0.7534588 0.59793544 0.6597839 -0.6750731 0.5270191 -0.53681076 0.45296547 -0.50098085 -0.68345565 0.71181095 0.7404297 0.45215493 -0.7751334 0.540177 0.6548013 -0.5523044 -0.40829453 0.4507893 0.6206888 -0.76933634 -0.47635654 0.4551281 0.5135999 0.5863066 -0.38903502 0.40588 -0.5274096 -0.4151696 -0.32737184 0.6121396 0.4612579 0.45288435 0.41017446 -0.50019133 0.40008807 -0.3322062 0.71391034 0.7309995 -0.67778075 -0.7206478 0.6000884; 0.61942333 0.49622065 -0.5628974 0.43168208 0.42561704 -0.79360604 -0.5890731 0.65104353 1.7722883 0.72354716 -0.5113287 -0.7407578 0.5099097 -0.53531265 -0.64406073 -0.6906313 0.42191947 -0.6251269 0.6694708 -0.5432049 -0.5125444 0.80758226 0.7024857 -0.50016767 0.6897111 0.79161566 0.45986748 -0.476394 0.69114035 0.51552826 -0.681247 -0.601367 0.4697123 -0.6601464 0.4435449 -0.60264176 0.8145688 -0.5444342 0.50822705 -0.39537722 -0.39702767 -0.6025426 0.6898841 -0.43976334 0.5758387 -0.6531576 0.55327344 0.3360291 -0.4520113 -0.6385849 -0.64298266 0.67693913 -0.48333156 -0.7101441 0.5434084 0.46679497 -0.4974636 -0.4202435 0.6215205 0.623467 -0.68551195 -0.4055439 -0.66418016 0.6214523 -1.1193813 -0.90807134 0.5087515 0.79965484 -0.42706 -0.7941664 -0.76638705 -0.4882165 0.6036389 -0.9184587 0.9789775 -0.39205337 -0.757643 0.6160078 0.57682025 -0.7447085; 0.59096956 0.47400585 -0.59703314 0.5823182 0.54929 -0.6864867 -0.80210197 0.65787464 -0.33818603 0.46458513 -0.7976673 -0.7220557 0.58458763 -0.46359774 -0.7013737 -0.34087119 0.39277107 -0.39678448 0.50998026 -0.35892615 -0.38430122 0.5790914 0.9503287 -0.3678997 0.4849998 0.5775641 0.682539 -0.60543036 0.44188455 0.54740924 -0.6937405 -0.44005585 0.87586236 -0.736539 0.56441885 -0.4032761 0.8878252 -0.73852015 0.564956 -0.6020114 -0.39522797 -0.6980017 0.64220583 -0.34197673 0.49747387 -0.36014163 0.5135272 -0.21292335 -0.39139187 -0.76444787 -0.7625417 0.39353737 -0.379908 -0.34160143 0.75037545 0.5723321 -0.20132089 -0.705052 0.62714976 0.556866 -0.69006157 -0.64202416 -0.68475735 0.6954235 -1.0339645 0.22876677 0.4908735 0.6911858 -0.47850642 -0.5210447 -0.6359347 -0.42646497 0.41586623 -0.65346193 -0.044473063 -0.5773582 -0.5609131 0.44667804 0.28222203 -0.59324974; -0.6745513 -0.58000773 0.58104664 -0.51610357 -0.7158398 0.7291938 0.46039012 -0.6056709 -0.610591 -0.5877051 0.63613594 0.55907613 -0.63313687 0.44830024 0.43419525 0.7545319 -0.5576653 0.5910255 -0.47228312 0.5042095 0.4633078 -0.5714442 -0.549153 0.62921995 -0.67425585 -0.5657029 -0.41599217 0.79577243 -0.4522471 -0.4962591 0.44304508 0.46598125 -0.776979 0.46936443 -0.44608155 0.5182332 -0.46521822 0.7193785 -0.4527078 0.7050883 0.5539148 0.7833617 -0.5721655 0.64719135 -0.72578776 0.4889368 -0.4010287 -0.5704834 0.6728783 0.5582562 0.6639292 -0.5168833 0.69319516 0.5188777 -0.6246104 -0.72924924 0.4956083 0.56833225 -0.55652064 -0.7110502 0.5043108 0.6030114 0.53724617 -0.4835794 0.6031266 0.6552778 -0.43730742 -0.38821095 0.66628677 0.5046419 0.6321356 0.7583476 -0.680811 -0.79061264 -0.69279414 0.7069514 0.6179418 -0.789721 -0.6752959 0.60726887; -0.48209357 -0.51174027 0.75218946 -0.64265484 -0.5963367 0.6978991 -0.8527759 -0.6929976 -0.27546218 -0.53983796 0.6333991 0.4889144 -0.46851593 0.55522066 0.37326413 0.60470515 -0.76730084 0.57249767 -0.4359572 0.57770854 0.41756073 -0.6609233 -0.61571276 0.5778842 -0.50139445 -0.7650926 -0.3906187 0.6460337 -0.65604734 -0.8184661 0.7879073 0.49966213 -0.7284352 0.47964984 -0.43044654 0.6507269 -0.50958824 0.6373976 -0.574715 0.5171677 0.74788713 0.5989364 -0.48620257 0.56238014 -0.6498854 0.70733356 -0.6303053 -0.63527507 0.6922863 0.46792492 0.37154725 -0.44128117 0.53667843 0.6340301 -0.3929297 -0.6599969 0.6944392 0.51468545 -0.36089885 -0.787434 0.7621899 0.49687278 0.44284108 -0.5097604 0.50618356 0.52017367 -0.57549036 -0.48557162 0.72506434 0.72762305 0.491661 0.39282864 -0.5439084 -0.78702456 -0.5125093 0.61036384 0.752146 -0.79045844 -0.7518161 0.47488484; 0.4664707 0.65823436 -0.48362318 0.6495483 0.4028175 -0.6705781 -0.8096836 0.53193957 1.0336323 0.44390604 -0.67155975 -0.43383008 0.7978124 -0.6070628 -0.6704819 -0.07149416 0.7004163 -0.45562938 0.4810378 -0.5479988 -0.5618469 0.7719393 0.67281806 -0.3368837 0.6622872 0.47292554 0.45070574 -0.65862566 0.5896876 0.47918054 -0.38220134 -0.46582618 0.85817605 -0.49180937 0.5115411 -0.38801894 0.6392935 -0.4942624 0.491642 -0.4688254 -0.629993 -0.69217336 0.76315 -0.52558 0.47927886 -0.39490864 0.48482132 0.35211292 -0.43274754 -0.54055536 -0.7123412 0.5784325 -0.52862394 -0.7308068 0.7914064 0.7035858 -0.38252065 -0.6203462 0.79688513 0.51202565 -0.6799842 -0.48225665 -0.4739588 0.87256473 -1.0059431 -0.9965759 0.6421165 0.75067055 -0.51672864 -0.77070606 -0.5793504 -0.43218172 0.63555527 -0.123001836 0.6315761 -0.49131432 -0.6428867 0.58223253 0.5214546 -0.43271628; 0.7214122 0.4819551 -0.72998 0.5614962 0.52329576 -0.7574829 -0.7196869 0.55853057 0.8107594 0.4471589 -0.6754471 -0.5595906 0.5681039 -0.47606796 -0.7948965 -0.5807788 0.7094357 -0.49826097 0.62459433 -0.5599366 -0.6001057 0.5411587 0.7052174 -0.62521553 0.5796635 0.43378383 0.71010387 -0.7742128 0.57844335 0.64825773 -0.44397894 -0.53027487 0.57204825 -0.7495084 0.62979525 -0.4971409 0.59129846 -0.5394655 0.42477337 -0.6672885 -0.4961729 -0.69531554 0.5538254 -0.46921995 0.54296744 -0.43513003 0.5989421 0.69807297 -0.5026089 -0.6340746 -0.77838624 0.444566 -0.43214563 -0.6867833 0.7813348 0.5689059 -0.68094015 -0.60925144 0.69790196 0.4130124 -0.6242827 -0.5999699 -0.7835298 0.58139545 -0.6224029 -1.0031064 0.5124224 0.74019367 -0.4035803 -0.49558952 -0.4296816 -0.6596553 0.5419902 -0.89667445 0.80407655 -0.6873551 -0.76030356 0.6532004 0.63717765 -0.5759588; -0.50632286 -0.40437186 0.62101275 -0.60993564 -0.48037693 0.5674763 -0.8065532 -0.6174039 -0.3639827 -0.7296745 0.4749684 0.5225314 -0.7622427 0.61479765 0.512194 0.81800324 -0.69901067 0.5254362 -0.55689114 0.62599754 0.66954345 -0.55191445 -0.44867975 0.76850164 -0.59201086 -0.43607774 -0.58737046 0.7853182 -0.5490321 -0.7731192 0.72854745 0.56839037 -0.43966272 0.6821905 -0.6832484 0.5604019 -0.6519177 0.66399175 -0.6486318 0.51017183 0.44727406 0.7629394 -0.44570985 0.66032004 -0.42864186 0.7374437 -0.4761001 -0.7581998 0.4459573 0.53146136 0.64710134 -0.7007991 0.45115128 0.4602167 -0.47536585 -0.7543778 0.7707405 0.6841344 -0.58018553 -0.64272463 0.7141431 0.557354 0.5083731 -0.49861118 0.58230925 0.65325934 -0.521325 -0.7695676 0.5698032 0.7856748 0.6023623 0.59744275 -0.43086725 -0.49650213 -0.63989305 0.509364 0.70277905 -0.6174264 -0.48045957 0.54999655; -0.83024 -0.740298 0.4652294 -0.38652435 -0.70225996 0.85368484 0.68749094 -0.56354547 -0.720907 -0.39105922 0.67866546 0.6847285 -0.85986704 0.5904435 0.7452911 0.22657163 -0.6411914 0.73199064 -0.758685 0.47904563 0.6266261 -0.5360364 -0.8184398 0.6244193 -0.47202882 -0.614386 -0.49480495 0.70576763 -0.7223538 -0.6032804 0.6159388 0.82157636 -0.50796366 0.73446214 -0.5560598 0.7313159 -0.62775344 0.47329527 -0.7691638 0.5953437 0.6827272 0.6677222 -0.6782118 0.46898437 -0.50661945 0.3907729 -0.524244 -0.4614391 0.6224958 0.50590503 0.85887164 -0.62179416 0.4128648 0.6232071 -0.51570886 -0.67226547 0.47779992 0.71473753 -0.7075935 -0.7470778 0.597055 0.59635013 0.7065354 -0.8189237 0.53142416 0.69246995 -0.66679496 -0.8440275 0.6295064 0.74021095 0.7053887 0.4795261 -0.53734374 -0.26066247 -0.8284397 0.53993064 0.7590186 -0.55979186 -0.50210714 0.4516892; 0.49065217 0.48613998 -0.7002379 0.59813 0.50346386 -0.52391493 0.76085424 0.6242777 0.58461565 0.4286667 -0.63976514 -0.46118084 0.5435919 -0.5690564 -0.42386663 -0.5892952 0.45528412 -0.50180054 0.6843633 -0.57354575 -0.57764834 0.75373214 0.7177827 -0.45001027 0.6707003 0.78413194 0.5482698 -0.6268786 0.51220644 0.7452758 -0.46601117 -0.5923882 0.61229056 -0.5799067 0.61254346 -0.58103013 0.6560018 -0.551843 0.5629053 -0.8012179 -0.5693314 -0.7933985 0.644498 -0.7493576 0.5095486 -0.5277079 0.5155734 0.4300326 -0.4909178 -0.534251 -0.6772959 0.48733267 -0.7676964 -0.6681704 0.6816446 0.58416814 -0.5029238 -0.48920488 0.53671616 0.47807387 -0.639469 -0.5150009 -0.46441168 0.5938899 -0.6130693 -0.43063214 0.47925273 0.6937427 -0.51406306 -0.5306954 -0.52150565 -0.4017058 0.5050579 0.60336256 0.688458 -0.61454666 -0.71842396 0.43793857 0.7089909 -0.58817273; -0.62029016 -0.33219215 0.52640873 -0.4844251 -0.7287135 0.4481844 0.75659645 -0.47419637 0.7694517 -0.73788124 0.42967564 0.7122075 -0.65627104 0.47495452 0.6422096 0.64935786 -0.7942693 0.42173076 -0.39736393 0.6439764 0.612575 -0.73381245 0.009637552 0.46361718 -0.65926147 -0.45499688 -0.555295 0.47341385 -0.594122 -0.56458765 0.8009968 0.5828742 -0.53658926 0.7549082 -0.71792793 0.77052325 -0.44325164 0.6447875 -0.44287625 0.6233117 0.48240262 0.63404673 -0.55331856 0.7627318 -0.6574779 0.749986 -0.4521131 0.35910943 0.81753385 0.54781395 0.74326533 -0.71288764 0.79501575 0.83148384 -0.6937917 -0.64089465 0.90994424 0.57596177 -0.42293057 -0.65160173 0.59797233 0.63351667 0.46902335 -0.021082992 0.026902443 -0.7174447 -0.49004626 -0.2507512 0.6876216 0.50686073 0.77003664 0.5944123 -0.7274258 0.38767403 0.82439315 0.6050872 0.40258613 -0.65949 0.014089722 0.39386913; -0.46538654 -0.7474952 0.50215757 -0.5379462 -0.41805527 0.54212815 0.8084596 -0.8194928 -0.5989632 -0.4265974 0.61670727 0.7423629 -0.84880275 0.5753655 0.60553575 0.20011984 -0.42861453 0.43829787 -0.4246067 0.5926322 0.39206263 -0.48196182 -0.7135774 0.35018718 -0.4841978 -0.5188887 -0.50894415 0.48247454 -0.43874416 -0.66551244 0.45618227 0.6538376 -0.79733646 0.39123133 -0.42479238 0.36380973 -0.4888676 0.650612 -0.5539113 0.63135964 0.6150289 0.75180596 -0.7040219 0.7465314 -0.42244995 0.56116235 -0.5483573 -0.4113586 0.32532325 0.56370133 0.664249 -0.66309 0.42583188 0.49132112 -0.67180175 -0.5401293 0.56993836 0.49084604 -0.78184944 -0.3914718 0.7043877 0.4363801 0.76621187 -0.7808593 0.7749118 0.7564344 -0.47461176 -0.74522793 0.42408872 0.667684 0.536273 0.6071316 -0.65442705 -0.12988184 -0.8410322 0.65463567 0.60093135 -0.6088836 -0.4005221 0.47573683; -0.66971946 -0.6090607 0.54922473 -0.7303795 -0.47338903 0.5125141 -0.72339034 -0.7603939 -0.40247104 -0.63216484 0.6137959 0.552221 -0.5919726 0.5339176 0.44307798 -0.564379 -0.50612146 0.48076558 -0.6214665 0.55491394 0.45663822 -0.51048005 -0.54392505 0.13161758 -0.58944184 -0.69329196 -0.5087825 0.45590904 -0.41577658 -0.6140736 0.43403646 0.7305942 -0.78936267 0.6029984 -0.7600261 0.636417 -0.4643187 0.5473634 -0.55077326 0.5567338 0.51302826 0.6189209 -0.7105733 0.7632351 -0.49455133 0.56687844 -0.5682596 -0.7175265 -0.4281698 0.40619215 0.39444765 -0.6848413 0.51723874 0.6080103 -0.55461454 -0.4041731 0.49499655 0.40018815 -0.78634846 -0.73247993 0.5930992 0.57701844 0.49064544 -0.4784379 0.57674164 0.5804994 -0.7364314 -0.44834653 0.5014189 0.78207564 0.63979113 0.49695155 -0.4659268 -0.63348436 -0.4071763 0.6275458 0.55460906 -0.50840753 -0.68903184 0.47289965; -0.76577705 -0.8195075 0.573511 -0.6554377 -0.57276714 0.9762043 0.69720584 -0.78069997 0.20318279 -0.48501918 0.76675427 0.56213874 -0.82842183 0.6214545 0.60245544 0.42967144 -0.54983705 0.4645972 -0.53055936 0.5038721 0.60137635 -0.5063332 0.29546937 0.49496126 -0.54696107 -0.6046419 -0.77404386 0.6834851 -0.77785265 -0.6617472 0.45289445 0.3738246 -0.8367627 0.7330458 -0.39708298 0.5117967 -0.7359909 0.49644986 -0.3512208 0.6447642 0.58763206 0.45741802 -0.70675814 0.34634262 -0.5997673 0.6767336 -0.49765033 0.9879474 0.43610066 0.57108694 0.55014116 -0.29180154 0.47830507 0.4255936 -0.51563704 -0.6272171 0.28075087 0.48587865 -0.780881 -0.6441195 -0.36305794 0.40231442 0.53772104 0.18941598 0.22165681 -0.6470741 -0.58335674 -0.8245004 0.6405468 0.7891577 0.5692505 0.8541557 -0.57801676 0.75130033 0.2559805 0.47342536 0.8257567 -0.45841628 0.77155507 0.64685625; 0.37203395 0.3800591 0.6659873 -0.521475 -0.47590116 0.7143807 -0.40081978 0.633878 -0.5805783 -0.6614499 0.9161025 -0.7428851 0.34885922 0.4086425 0.6066312 -0.66270655 -0.4652806 0.32072955 0.34757936 0.5835681 0.2985995 0.54609674 -0.93802696 -0.6038706 0.10599276 -0.7189127 0.51077396 -0.8966634 -0.66812456 0.6654761 -0.8381616 -0.45137763 -0.13166724 0.3050837 -0.40253025 -0.7638094 -0.7610852 -0.7678534 -0.6964561 -1.0367782 0.5857131 0.6292593 -0.6679953 0.48111656 -0.5890031 -0.89877385 0.5597343 -0.40241158 -0.6736478 -0.36529484 -0.5246529 -0.51424694 0.241663 0.44475168 0.64607394 -0.6441074 -1.033179 -0.6899353 0.34220767 -0.47685835 0.38954085 0.6084502 -1.1042916 -0.86710465 1.1239494 0.5849524 -0.56198615 -0.63670087 -0.78871965 -0.6310225 0.51745284 0.5991063 0.92628676 -0.515988 -0.8858674 -0.10126705 0.6929905 -0.65526503 -0.5213002 0.8157927; -0.43734118 -0.5077678 0.42038825 -0.52315855 -0.40640458 0.63980216 -0.5488038 -0.6837354 -0.69043475 -0.47626346 0.4587846 0.6766766 -0.60084647 0.52069056 0.4264355 -0.44870958 -0.6089649 0.47936735 -0.59654045 0.52578354 0.6039474 -0.4277799 -0.71560264 -0.41503495 -0.70059127 -0.57013905 -0.814172 0.38445657 -0.61002785 -0.78010434 0.66479987 0.5412328 -0.46750394 0.4747486 -0.73549646 0.59177506 -0.7981103 0.41931352 -0.5371042 0.68202025 0.67047745 0.46333373 -0.5062155 0.5094251 -0.51584226 0.5155834 -0.60953456 -0.63255346 -0.34759527 0.70912576 0.263638 -0.7271036 0.43494138 0.69662946 -0.53630185 -0.67869246 0.6928459 0.58193946 -0.617185 -0.56302947 0.6949611 0.6826437 0.41848934 -0.5686343 0.7362492 0.6526848 -0.434942 -0.74541694 0.59145796 0.4726356 0.7476288 0.58984756 -0.6478281 -0.4219197 -0.72647697 0.49753475 0.6414108 -0.75282407 -0.50926846 0.6571989; -0.3256292 -0.59740955 0.7471547 -0.6227965 -0.5890152 0.43520412 0.68957585 -0.31452787 -0.72773236 -0.44725278 0.53544194 0.587233 -0.650793 0.52421767 0.739669 0.6409266 -0.69411737 0.7171933 -0.3214402 0.4832657 0.66022605 -0.5007861 -0.5523844 0.4415381 -0.73152596 -0.99852216 -0.69251966 0.5357698 -0.71563786 -0.8248848 0.69661367 0.2697097 -0.42515665 0.698798 -0.47796246 0.72559506 -0.63234836 0.554884 -0.49406412 0.75878865 0.452557 0.48084542 -0.64173377 0.56781286 -0.6446895 0.7852461 -0.54392767 -0.45218983 0.78670686 0.4331415 0.52853584 -0.50015795 0.68395406 0.61754125 -0.5277463 -0.627214 0.53696203 0.43315476 -0.5595741 -0.49955133 0.6896432 0.63910115 0.41259217 -0.75286865 0.56883967 0.55908793 -0.6412179 -0.37695923 0.57158816 0.6096569 0.7741605 0.7596697 -0.76554644 -0.62702394 -0.541235 0.6011298 0.5941162 -0.6811265 -0.58829373 0.49701685; -0.6269831 -0.5556916 -0.6745731 0.80519414 0.710957 -0.43941373 0.7130705 -0.6341379 0.49444047 0.78572065 -0.714085 0.3413989 -0.6477024 -0.6914322 -0.48301914 0.673794 0.56172985 -0.66705704 -0.41803142 -0.7132255 -0.55337256 -0.53791416 0.643818 0.3885683 -0.18228732 0.5244801 -0.68190706 0.6303183 0.65379506 -0.63386476 0.42082503 0.41286558 -0.7984144 -0.9005704 0.6709418 0.62306863 0.6713945 0.49033397 0.5370536 0.28636557 -0.6765141 -0.7684068 -0.05825204 -0.48570922 0.86846185 0.26398954 -0.6438769 0.50071704 0.70299107 0.54937434 0.8152203 0.69094974 -0.54425395 -0.6455412 -0.76493233 0.59390295 0.29285148 0.5923598 -0.7514045 0.58929527 -0.5770705 -0.781265 0.40135422 0.678998 -0.52899545 -0.56660646 0.42651606 0.2948956 0.32185188 0.45273924 -0.57076097 -0.36391443 -0.66062707 0.79029185 0.6293686 -0.5781953 -0.5142699 0.5526436 0.6921372 -0.44643006; 0.5336149 0.72549516 -0.5205889 0.44545338 0.7139149 -0.5340118 -0.6933295 0.6747601 0.6306392 0.8836637 -0.72451323 -0.58252084 0.6325262 -0.7649915 -0.44102773 -0.42342043 0.5357488 -0.47674438 0.5994745 -0.43492985 -0.7612207 0.45450228 0.45801723 -0.7735463 0.48363346 0.7551074 0.8065867 -0.6834731 0.63807803 0.42943332 -0.69103587 -0.31495866 0.6566236 -0.44726354 0.47358677 -0.7279392 0.63201463 -0.79209757 0.5688966 -0.5811356 -0.470071 -0.6469155 0.59612983 -0.6722723 0.64576864 -0.49843842 0.712183 0.44671792 -0.76847935 -0.62848276 -0.5254077 0.42655015 -0.5178065 -0.42591736 0.5847868 0.67585576 -0.6969843 -0.7153506 0.70908964 0.4288087 -0.42844948 -0.4357473 -0.47792286 0.6126084 -0.5513726 -0.58367085 0.40124393 0.50187147 -0.6535274 -0.7639235 -0.7640401 -0.7299561 0.67939633 0.49303263 0.4831174 -0.615994 -0.68205553 0.5218276 0.77079743 -0.52042323; 0.6206756 0.4237957 -0.5435097 0.64337134 0.6738142 -0.68648165 0.8635926 0.42343518 0.37039787 0.70999485 -0.6491232 -0.59579784 0.5371621 -0.75539356 -0.5378181 0.49258655 0.45015848 -0.6396354 0.53732735 -0.7979576 -0.4788648 0.706391 0.5616928 -0.5166536 0.41460592 0.7206126 0.70264 -0.7009224 0.43881288 0.7152554 -0.58959204 -0.7938178 0.52112454 -0.52279526 0.5688417 -0.66372776 0.66178584 -0.65473074 0.56151193 -0.6134564 -0.70224714 -0.80331004 0.695392 -0.73628885 0.7294645 -0.8160505 0.6145742 0.5727296 -0.50277025 -0.37727806 -0.37953335 0.7230194 -0.5358892 -0.64933544 0.63191074 0.7431 -0.5513294 -0.65545034 0.4918731 0.8116422 -0.5913731 -0.47571045 -0.60543096 0.6602874 -0.5076091 -0.58431363 0.40499675 0.36147818 -0.75247395 -0.43285143 -0.45877612 -0.5925337 0.47568086 0.7666221 0.43132374 -0.5389369 -0.3825991 0.720544 0.5716714 -0.4176383; 0.7860998 0.5387374 -0.5394287 0.39967462 -0.0018672248 0.60912055 -0.5649899 0.64846903 -0.5850557 0.93179375 0.34217113 -0.61928135 0.3855464 0.4623265 0.2546897 -0.70551556 0.622596 -0.6300714 0.6759717 0.4806935 -0.58716476 0.59950155 -0.9305349 -0.7017988 0.6555702 -0.0889424 0.79122376 -0.8098709 0.44597295 0.42609945 -0.5630137 -0.6231043 0.109322146 -0.5720819 -0.23710118 -0.4868771 -0.38813305 -0.60720927 -0.6980391 -0.82139224 -0.6422888 0.016325107 0.5078532 0.36687848 -0.34164926 -0.650809 0.66607785 -0.5689191 -0.746223 -0.77503175 -0.52798355 -0.51819974 -0.70507526 0.6596997 0.5969593 0.705808 -0.22645706 -0.8090512 0.19346501 -0.24804702 0.7172978 -0.80263627 -0.61429805 -0.6358635 0.6901012 0.6160865 -0.31801358 -0.35170716 -0.44361383 -0.8094079 0.17489456 -0.54373455 0.59283304 -0.47015548 -0.585883 -0.17139551 -0.07723202 0.68510795 -0.7161292 -0.7373671; -0.5127764 -0.50415003 0.777092 -0.8002368 -0.7904714 0.63359845 -0.57868224 -0.49404967 -0.45711347 -0.6679481 0.5794208 0.56445855 -0.5671451 0.61952513 0.49053493 0.7269119 -0.5716165 0.79764676 -0.50330025 0.4978964 0.59268713 -0.5082966 -0.5269251 0.69037235 -0.47054356 -0.7195065 -0.5934408 0.7833694 -0.6414179 -0.6749135 0.69341385 0.77557987 -0.60507935 0.56982565 -0.5598162 0.70252764 -0.52286524 0.486591 -0.46798766 0.5647136 0.45616752 0.63305235 -0.44022447 0.42167807 -0.43039972 0.6442126 -0.50568384 -0.57960856 0.5585562 0.6303105 0.7240517 -0.5673514 0.4761252 0.4304055 -0.6293658 -0.55165344 0.7534785 0.51176167 -0.67757654 -0.4353996 0.47678185 0.46924642 0.5011337 -0.6591687 0.74105906 0.52641594 -0.5066529 -0.43892202 0.6193803 0.5811268 0.6393096 0.49285445 -0.5700444 -0.49638814 -0.4930475 0.64412946 0.61167824 -0.4849594 -0.77121556 0.75678176; -0.75376207 -0.8966898 0.5275345 -0.51209396 -0.5254937 0.5247283 -0.047026176 -0.8036509 -0.71266603 -0.6707316 0.65799695 0.58722806 -0.6045119 0.40282035 0.752846 0.32752854 -0.6629216 0.6724711 -0.82289594 0.34959838 0.50439835 -0.6518203 -0.80158824 0.3850346 -0.57710034 -0.3743905 -0.5772376 0.657969 -0.6004742 -0.55741745 0.6359751 0.39027187 -0.50174445 0.6615939 -0.43127176 0.40215066 -0.58487535 0.573307 -0.631665 0.6571052 0.70561713 0.5064193 -0.85036165 0.6709305 -0.44609645 0.40983826 -0.75297743 -0.26637986 0.32023445 0.8291312 0.8756654 -0.46187013 0.60727435 0.3492483 -0.66341007 -0.76336336 0.46531332 0.75546324 -0.48108426 -0.52407277 0.6115413 0.5999173 0.7082211 -0.86129457 0.62215716 0.5968989 -0.5153176 -0.83762705 0.23994043 0.7189201 0.5565941 0.69242036 -0.5953176 -0.3824795 -0.60208225 0.5047145 0.78960025 -0.45969784 -0.6156812 0.54292446; -0.83962226 -0.775234 0.53090394 -0.64379597 -0.7496818 0.5211383 -0.66024965 0.58236706 -0.73633456 -0.4373698 0.7751945 0.45593333 -1.0855535 0.5577032 0.8107287 -0.70958406 -0.46564788 0.5031298 -0.52255726 0.670325 0.61977386 -1.0137892 -0.61850893 -0.64395064 -0.5272354 -0.6102073 -0.44956625 -0.66761583 -0.65271014 -0.56025606 -0.5465186 -0.6823203 -1.0199336 0.3772799 -0.498934 -0.6381646 -0.8750428 -0.93861073 -0.38874766 0.38115144 0.6633954 0.70230585 -0.7619169 0.6347265 -0.53991264 0.5317427 0.590619 -0.61922514 -0.82444406 0.7680879 -0.45940962 -0.46287468 0.4191507 0.6004905 -1.0602949 -0.71550924 -0.2397119 0.7237746 0.28344664 -0.46771315 0.75662 0.56749344 0.38124517 -0.7814321 0.9141168 0.6022665 -0.8329279 -0.6228526 -0.23849204 0.7821702 0.41270325 0.7171021 -0.28692657 -0.5799676 -0.7128502 0.22135545 0.5954131 -0.3768272 -0.39692762 0.57888246; 0.4822085 0.45972836 0.62527806 -0.5821964 -0.5147785 0.49766982 -0.4707498 0.8078812 -0.75554985 -0.4082643 0.5177528 -0.6702241 0.56551385 0.5696012 0.7919341 -0.62989914 -0.36642876 0.3954638 0.67711264 0.5060438 0.33357078 0.5774011 -0.64362025 -0.7883692 0.5918987 -0.4243205 0.709949 -0.56847924 -0.57159746 0.46520847 -0.77367663 -0.4475743 0.31209266 0.33220658 -0.6263864 -0.7648792 -0.64695376 -0.4572223 -0.62890947 -0.82805425 0.43180618 0.55471075 0.4718204 0.5876031 -0.72176534 -0.5289495 0.7826117 -0.58695483 -0.5253091 -0.57087857 -0.60626775 -0.48285195 0.28068343 0.7606701 0.63744277 -0.568949 -0.48015136 -0.77941966 0.69315726 -0.40000314 0.5983784 0.31538802 -0.566899 -0.7795715 1.0061516 0.774473 -0.7474761 -0.7436087 -0.56029886 -0.68161386 0.57601184 -0.6215531 0.562799 -0.69423586 -0.6952953 -0.025341287 0.48925138 -0.6193298 -0.75588286 -0.7308614; 0.7049856 0.50684834 -0.48748982 0.76951045 0.4879328 -0.61359024 0.7210018 0.7652212 0.76269203 0.65270835 -0.51956326 -0.5004331 0.75390565 -0.6009736 -0.7591762 0.51515526 0.48749378 -0.6018345 0.70188886 -0.6239977 -0.39364526 0.47740167 0.47698376 0.8348205 0.71135217 0.68815696 0.6067235 -0.67926204 0.74337053 0.5409499 -0.62783957 -0.3936864 0.6147608 -0.42652747 0.76819724 -0.20233303 0.46420628 -0.41464517 0.6932863 -0.4119917 -0.75824714 -0.5959629 0.77622813 -0.44699714 0.48303983 -0.6938038 0.7146787 0.667746 0.83822626 -0.6101595 0.91059875 0.50446016 -0.5126803 -0.4618018 0.46643817 0.42371547 -0.6770811 -0.76005286 0.38914907 0.786182 -0.789878 -0.46552882 -0.5220594 0.45398366 -0.73636824 -0.57919264 0.5160369 0.49753347 -0.47986493 -0.55615675 -0.4107172 -0.440867 0.3902917 0.815996 0.7557352 -0.4449145 -0.612834 0.5389455 0.55503106 -0.76135856; -0.64216346 -0.8389068 0.40317318 -0.39682814 -0.3562835 0.78851086 -0.51547 0.60323036 -0.55229634 -0.5969659 0.76059085 0.7430133 -0.68041664 0.57884717 0.79869986 -0.6334633 -0.61811537 0.4112284 -0.5117942 0.37736958 0.47538677 -0.65516514 -0.6920885 -0.7873822 -0.8379712 -0.71124125 -0.6058183 -0.6334448 -0.6600754 -0.4630395 0.38212737 -0.51846164 -0.8383353 0.5256988 -0.70748943 -0.7643951 -0.7094226 -0.77777755 -0.72104526 0.3431196 0.6822566 0.61590093 -0.62015235 0.4921948 -0.60161006 0.57496274 0.6283664 -0.6977826 -0.8693401 0.43468654 -0.3706218 -0.72495717 0.3957871 0.689566 -0.5810805 -0.8112836 0.39802116 0.5155054 0.34625882 -0.58059615 0.48338133 0.54008806 0.4024534 -0.63909525 0.85669005 0.7777868 -0.87877434 -0.6760142 0.42048672 0.6693239 0.37008005 0.58790237 -0.6509182 -0.57152927 -0.56771326 0.5612506 0.83537596 -0.6619847 -0.34529725 0.6243047; -0.5454341 -0.68477017 0.41643155 -0.6918433 -0.5016444 0.8173213 0.65706956 -0.76902646 -0.7266719 -0.6764648 0.65322423 0.5076082 -0.5539801 0.6053936 0.6873229 0.6370935 -0.7107421 0.6426438 -0.5679547 0.6931643 0.72815853 -0.7182567 -0.6859825 0.37129357 -0.54674476 -0.725075 -0.59907705 0.56826997 -0.7687688 -0.57977957 0.41637433 0.42483687 -0.7603389 0.43733916 -0.57480645 0.63868517 -0.64025337 0.47001478 -0.62413514 0.5694584 0.38479725 0.5082547 -0.8317089 0.6159444 -0.36787996 0.73475194 -0.5318609 -0.31233704 0.65684456 0.8258865 0.77862746 -0.4760503 0.4497159 0.4417349 -0.55134195 -0.58083075 0.33374047 0.6843092 -0.44561613 -0.48665386 0.4671512 0.6182985 0.39961496 -0.70248085 0.34589446 0.51501113 -0.6892759 -0.70902944 0.42435673 0.7729213 0.6498387 0.50765115 -0.47525832 -0.41605547 -0.56048745 0.627802 0.7752253 -0.710806 -0.3803207 0.7933202; 0.83967113 0.4344132 -0.79576296 0.54082066 0.58761674 -0.6900816 0.5009974 0.3992059 0.3721246 0.7394721 -0.7422845 -0.804033 0.6704688 -0.5373056 -0.58320725 0.5289109 0.77563775 -0.7404268 0.50841004 -0.80794317 -0.7095386 0.63486725 0.6126188 0.28236467 0.4118257 0.7229259 0.6322768 -0.4422193 0.52533996 0.5376667 -0.47194716 0.6879952 0.35347408 -0.7160177 0.7486735 0.21339898 0.72260773 0.2508594 0.5618848 -0.78567994 -0.7107278 -0.5973578 0.51212585 -0.6849287 0.7991793 -0.746271 -0.52132815 0.54439884 0.29513943 -0.7772831 0.7610293 0.8147224 -0.54069126 -0.68951607 0.41436324 0.62204236 -0.37307388 -0.6188209 -0.6495196 0.6639564 -0.5000665 -0.8206516 -0.46563932 0.38569131 -0.40018967 -0.3988971 0.45808655 0.63741857 -0.5399215 -0.53471005 -0.7233421 -0.47045067 0.6453996 0.46957356 0.47939447 -0.67630035 -0.67421055 0.8204994 0.5511392 -0.7206184; -0.4340782 -0.5349899 -0.5500053 0.49321395 0.48497412 -0.7544302 0.49099892 -0.77600515 0.45175448 0.7242699 -0.61086094 -0.18233193 -0.6909842 -0.44488147 -0.76705146 0.6196528 0.8249041 -0.5583899 -0.26011902 -0.7192403 -0.79914 -0.61654544 0.46466565 0.7420048 0.4404532 0.69740397 -0.59797215 0.7086282 0.7596548 -0.47600007 0.5725916 0.5310623 -0.5578226 -0.7747884 0.7350841 0.43039316 0.4441819 0.6313614 0.5196114 -0.7667882 -0.6162526 -0.78360045 0.6517497 -0.6394867 0.46101892 -0.52052844 -0.6338583 0.50126725 0.7586726 0.4697608 0.66080374 0.7617537 -0.7903225 -0.8106478 -0.6472684 0.53973055 0.20604816 0.63193 -0.5264196 0.6618646 -0.7038599 -0.78867394 -0.8039226 0.7564839 -0.40033457 -0.5960599 0.5183559 0.5280347 0.5646644 -0.21023948 -0.5774246 -0.55779874 -0.4466286 0.66538894 0.39455685 -0.51645905 -0.4966294 0.62425655 0.45343423 -0.66046375; -0.7312485 -0.44722697 0.4124215 -0.7864053 -0.70138556 0.41135365 0.41245127 -0.41695127 -0.36359525 -0.44523743 0.5639934 0.63820434 -0.35273024 0.42691824 0.39839798 0.81689775 -0.51327807 0.53323275 -0.50748944 0.48757878 0.4162546 -0.5414913 -0.669131 0.74646217 -0.6687411 -0.53012717 -0.46015376 0.513213 -0.7302518 -0.68701375 0.4576529 0.7117723 -0.74098974 0.49925584 -0.5612668 0.7829008 -0.4517706 0.7720417 -0.79744965 0.57935137 0.4392857 0.46203876 -0.49746862 0.5957672 -0.58615917 0.752403 -0.37240857 -0.6269328 0.60222554 0.4222235 0.49451554 -0.66510135 0.5405955 0.5104042 -0.39893445 -0.4292196 0.53338104 0.69108766 -0.49099866 -0.5335559 0.53176916 0.51997495 0.77876633 -0.61476094 0.5843133 0.4125887 -0.7056055 -0.5939199 0.5326744 0.55152404 0.62983704 0.50770015 -0.5211185 -0.11062224 -0.3448183 0.6151163 0.37389585 -0.66640353 -0.53934556 0.52422184; 0.7971517 0.7284243 -0.3793134 0.64533955 0.5221938 -0.6389367 0.3128096 0.6244318 0.5276578 0.6056112 -0.6439281 -0.72459024 0.5888427 -0.6766831 -0.7983304 0.70179325 0.42521533 -0.72954863 0.66833353 -0.41040552 -0.6320295 0.63749534 0.5867768 -0.45638415 0.6996922 0.5412892 0.810157 -0.34497628 0.56822526 0.4196226 -0.68219745 -0.6051468 0.54591745 -0.6249966 0.42951465 -0.47448793 0.74028903 -0.31206566 0.7216799 -0.63075787 -0.62661815 -0.6155138 0.6469649 -0.54174286 0.36457342 -0.44954813 0.7619231 0.3687116 -0.1828411 -0.6194765 -0.77416825 0.42572874 -0.66184986 -0.7087828 0.64219314 0.63159 -0.47111312 -0.7957765 0.5171107 0.6598675 -0.48599362 -0.7091581 -0.68477345 0.553138 -0.5306364 -0.6958528 0.57222617 0.712968 -0.30682763 -0.7708888 -0.39532453 -0.79841757 0.69627774 0.4911459 0.71557564 -0.34206825 -0.7489182 0.47792754 0.38770252 -0.54005563; -0.48691165 -0.5928374 0.82176876 -0.5866643 -0.7522013 0.7226902 0.67550564 -0.75580865 -0.51106197 -0.6807907 0.7233745 0.75071007 -0.5142018 0.668373 0.7264757 0.54631454 -0.63052773 0.8316858 -0.47968063 0.6970472 0.44627637 -0.53464043 -0.5434973 0.72806984 -0.671192 -0.64367056 -0.45336393 0.51373625 -0.48405188 -0.44677308 0.5924631 0.74092704 -0.48430473 0.5543798 -0.5090901 0.6166545 -0.6557031 0.71535754 -0.7620142 0.55983573 0.47275198 0.7800637 -0.6229193 0.57864934 -0.6233194 0.61284554 -0.7286421 -0.67774075 0.7758022 0.7346091 0.5666371 -0.7461933 0.57038665 0.5301521 -0.6786785 -0.72684926 0.56979007 0.50157714 -0.3598048 -0.6162861 0.67445165 0.8095527 0.5630806 -0.4784657 0.33551165 0.4565417 -0.7271942 -0.58771026 0.8010066 0.46140125 0.5260472 0.6280154 -0.83623487 -0.60232925 -0.51692384 0.71348774 0.6646532 -0.5631774 -0.4647685 0.48339137; -0.71504027 -0.7504879 0.78889656 -0.50683665 -0.4685402 0.67209095 -0.7001362 -0.6920745 -0.66368335 -0.47815824 0.5220397 0.6696617 -0.43649828 0.6154463 0.6694152 0.7022027 -0.48257193 0.41370502 -0.5735537 0.6958903 0.71283466 -0.4912697 -0.424924 0.7858691 -0.70893705 -0.45586827 -0.5940398 0.5937005 -0.6463546 -0.79863405 0.5913866 0.73058796 -0.5025844 0.5182427 -0.45973 0.5049384 -0.44458613 0.72830087 -0.52200407 0.69767666 0.60384536 0.54322714 -0.40069085 0.6312762 -0.45063668 0.4293108 -0.532839 -0.75969243 0.77190334 0.72892946 0.61295646 -0.7032945 0.51105535 0.44101906 -0.53378654 -0.51019275 0.67026085 0.72457427 -0.72304446 -0.6916803 0.75225097 0.4337173 0.62390655 -0.40273124 0.45572165 0.55299747 -0.6402552 -0.4954327 0.49264723 0.41103995 0.6098051 0.55232066 -0.6908483 -0.51712084 -0.5784129 0.5733845 0.77089864 -0.6630109 -0.5543869 0.57727545; 0.7580789 0.62233615 -0.7057492 0.5554984 0.5422337 -0.41657606 -0.43491793 0.69114995 0.7087565 0.5632241 -0.59034616 -0.5012847 0.72200286 -0.675507 -0.45387167 -0.44024622 0.7306208 -0.47240174 0.7712943 -0.5788502 -0.7379884 0.59014547 0.38235685 -0.4292011 0.7041627 0.6219481 0.69712955 -0.5305737 0.5513659 0.5121253 -0.58082914 -0.35994002 0.6675131 -0.71601343 0.47629473 -0.44548383 0.7347807 -0.46321794 0.6637674 -0.7302774 -0.5141862 -0.47171804 0.6981328 -0.81215125 0.54732054 -0.6741924 0.57348895 0.4600244 -0.5383339 -0.44002104 -0.5547134 0.6199049 -0.45289338 -0.5638376 0.757673 0.65064067 -0.50546646 -0.74156445 0.74328995 0.6548773 -0.69259626 -0.542593 -0.6032893 0.57179534 -0.6913488 -0.5216223 0.72428155 0.49350166 -0.5490228 -0.6765478 -0.66874313 -0.6094782 0.4690652 0.50099015 0.5969856 -0.66709334 -0.39767957 0.5648009 0.76841 -0.6365988; 0.63380605 0.78168905 -0.69165117 0.41844952 0.7523484 -0.6710128 -0.67770225 0.44519153 1.3589118 0.64601654 -0.49359706 -0.5859972 0.58350724 -0.68301433 -0.73036325 -0.44317296 0.5987742 -0.47663587 0.4135393 -0.4441346 -0.7266078 0.6921944 0.70867527 -0.54247046 0.54953736 0.5423205 0.723742 -0.5984167 0.796931 0.42457628 -0.47187626 -0.68068224 0.55823576 -0.48449317 0.69392216 -0.7326846 0.6662074 -0.6862705 0.4540485 -0.6667023 -0.5309582 -0.68591577 0.62015516 -0.3962892 0.67757815 -0.42214218 0.7596771 0.44159842 -0.60936946 -0.50941676 -0.6483797 0.4217181 -0.62615854 -0.41162705 0.47886986 0.6669037 -0.4610481 -0.59985775 0.57307214 0.64006853 -0.5783183 -0.39439526 -0.71318305 0.78903544 -0.7625757 -1.0628217 0.8079397 0.7820505 -0.44482684 -0.5010288 -0.49060762 -0.43880004 0.40690154 -0.9277505 0.69978124 -0.3789977 -0.54655033 0.63245267 0.42843947 -0.5150846; -0.5000729 -0.503757 0.49648598 -0.4483417 -0.6832004 0.6433202 0.56336135 -0.7885343 -0.9635626 -0.5939452 0.6563686 0.5509887 -0.76350844 0.7571581 0.4765343 0.74424565 -0.6214637 0.5201577 -0.5517582 0.7145634 0.7428801 -0.67929345 -0.64928794 0.5269025 -0.5211463 -0.5817776 -0.4659482 0.52437526 -0.69530565 -0.674584 0.6547672 0.3937056 -0.46361646 0.61667854 -0.46310326 0.5600894 -0.52204317 0.778186 -0.47073373 0.6257567 0.44174904 0.43196383 -0.7754877 0.6199441 -0.61290795 0.4195906 -0.6638152 -0.4714069 0.7041944 0.6612608 0.42540586 -0.4820058 0.7708341 0.7275206 -0.7312907 -0.66338944 0.40550125 0.52720934 -0.77001363 -0.46307173 0.5161281 0.5804809 0.5139424 -0.58341044 0.6961981 0.856945 -0.5235417 -0.5983906 0.67568505 0.7645516 0.4580695 0.729181 -0.6479228 0.5358215 -0.6371172 0.476951 0.6111707 -0.7163034 -0.6447589 0.74907964; -0.52398485 -0.58305115 0.42269084 -0.5265827 -0.6846155 0.58991164 0.6136561 -0.40973255 -0.44391447 -0.73174185 0.5378062 0.46703708 -0.6315645 0.4139219 0.6427442 0.7623511 -0.45741197 0.574518 -0.7682973 0.4765767 0.6614266 -0.523426 -0.56784254 0.46320885 -0.7495782 -0.4718401 -0.5394507 0.46064284 -0.6774641 -0.7772776 0.5447172 0.42397934 -0.6577807 0.79985476 -0.50860155 0.4436717 -0.7085041 0.46263853 -0.7676728 0.7740726 0.79228586 0.56123704 -0.6831003 0.7515264 -0.60753846 0.5047289 -0.6325037 -0.5289908 0.61529934 0.45384717 0.7563166 -0.6281764 0.6603722 0.46960965 -0.76384103 -0.5550552 0.7027017 0.7286322 -0.42932603 -0.5581986 0.45759848 0.51768464 0.57225466 -0.54788345 0.54158354 0.57800424 -0.746238 -0.6906975 0.46295738 0.71991247 0.44653687 0.51598173 -0.71861434 -0.65830857 -0.51955485 0.44934076 0.38442612 -0.43645406 -0.47256765 0.69074017; 0.917059 -0.10169804 0.4857732 -0.47415048 -0.6870833 0.4721726 -0.71464485 0.50665176 -0.41736317 -0.6420067 0.37236446 0.56370586 -0.008187012 0.6840501 0.37065184 -0.53655696 -0.4251053 0.6239254 -0.38716888 0.76058954 0.48783672 0.95335525 -0.42929804 -0.40724885 -0.2921312 -0.6088302 -0.5452197 -0.7051119 -0.3909222 -0.72032773 -0.36895546 -0.60768867 -0.6698809 0.6274881 -0.6554175 -0.6270954 -0.64023453 -0.3812337 -0.43407887 0.69825083 0.7841912 0.65246344 -0.57980347 0.72904706 -0.48425657 0.49957502 0.4926849 -0.64373344 -0.5987827 -0.8132315 -0.84762466 -0.73601156 0.7068926 0.54094654 0.7496648 -0.3741111 0.41539666 -1.0447956 0.5223491 -0.72250545 0.79592526 0.4493206 0.44268864 -0.51586294 0.66975385 0.40886497 -0.6843429 -0.49195105 0.0848465 0.5134439 0.49647048 0.453185 0.3361517 -0.5637984 -0.51035124 0.592418 0.62149364 -0.6672933 -0.6978412 0.45599574; 0.7847188 0.58345115 -0.4331031 0.4105285 0.6318096 -0.69699514 0.55455863 0.6132211 0.8736151 0.44796684 -0.78738266 -0.46020353 0.7576549 -0.6568246 -0.5356556 0.92386913 0.59496546 -0.53029555 0.6570504 -0.5856454 -0.5215288 0.48985162 0.8343387 -0.15522753 0.57318276 0.51586396 0.64566404 -0.36829796 0.57708484 0.62935287 -0.5329801 -0.6129692 0.6442242 -0.47100967 0.5181814 -0.45848224 0.8140387 -0.30673394 0.72142315 -0.48684248 -0.652437 -0.6177563 0.66281766 -0.4292063 0.40179798 -0.6028321 0.75684655 0.51197386 0.10862894 -0.5165161 -0.6946419 0.4499637 -0.6031614 -0.65222985 0.570134 0.6183652 -0.2571188 -0.68223697 1.0097954 0.49028796 -0.7049115 -0.49450633 -0.48770973 0.6291835 -0.76338977 -0.80358374 0.5787774 0.69513774 -0.30082306 -0.555624 -0.5008018 -0.42977193 0.5915291 0.44692034 0.68376905 -0.43371758 -0.50095576 0.66754216 0.637214 -0.7418445; 0.29782462 0.5062571 0.7789142 -0.7221175 -0.67551583 0.51975155 -0.68745166 0.637071 -0.5369806 -0.6320447 0.52972656 0.5415597 0.48846972 0.68660223 0.6280593 -0.52258927 -0.6611794 0.7552928 0.37158462 0.5078079 0.92664933 0.60896254 -0.33511415 -0.6395277 -0.30718103 -0.68134373 0.27804142 -0.45342413 -0.67459327 0.37028232 -0.11131887 -0.46897355 0.5679909 0.56629 -0.75858754 -0.6000754 -0.3026881 -0.53197896 -0.67648387 0.60597086 0.7557402 0.570423 -0.75715697 0.80962265 -0.75814307 0.6488385 0.40982395 -0.60064125 -0.39122307 -0.58513665 -0.4951723 -0.48727608 0.9084721 0.8386327 0.4253775 -0.67738646 0.16582264 -0.4668867 0.9046483 -0.5636804 0.70183617 0.7786848 0.90700406 -0.6757766 -0.042960282 0.33046234 -0.30889186 -0.30981168 -0.2204397 0.44582868 0.7639962 0.5602034 0.22571611 -0.48801303 -0.45362142 0.49930865 0.42873976 -0.85731196 -0.80897146 0.57180655; 0.6215585 0.7094423 -0.69668436 0.58978236 0.69286406 -0.6168151 -0.7096604 0.40529564 0.57301235 0.65900576 -0.43373585 -0.59624934 0.40596315 -0.44084007 -0.7056462 -0.59245557 0.49833015 -0.74590296 0.7194171 -0.48756507 -0.44391772 0.52973825 0.70339745 -0.72704434 0.46528953 0.5067541 0.4693304 -0.8008288 0.4795552 0.38248774 -0.48314723 -0.6408818 0.6472875 -0.64061666 0.6143155 -0.70507944 0.6371376 -0.5830036 0.7794278 -0.7415986 -0.5416596 -0.5816595 0.59400886 -0.61452574 0.55736214 -0.6533912 0.76138824 0.51044315 -0.7981034 -0.58082986 -0.73467827 0.660744 -0.61393464 -0.5352925 0.53653705 0.5345354 -0.69594413 -0.3916324 0.4162293 0.7583595 -0.7776415 -0.5001016 -0.59226906 0.70145744 -0.5398586 -0.65653706 0.7643193 0.71808887 -0.6306859 -0.6445081 -0.71679395 -0.74701715 0.6992556 0.6194732 0.74036777 -0.5052849 -0.75624883 0.6771303 0.5695436 -0.6913892; 0.7660145 0.73621005 -0.40648693 0.66963625 0.38623866 -0.7691219 0.7231926 0.71953875 0.54125816 0.87120867 -0.73002493 -0.77084506 0.7856981 -0.5861416 -0.5526238 0.61198944 0.51201 -0.4764393 0.8140351 -0.7052049 -0.80857736 0.62470496 0.4101183 0.41713473 0.6476974 0.77597123 0.5440984 -0.43770084 0.7818396 0.69145554 -0.5670688 0.36612824 0.58195794 -0.74036396 0.51735806 0.060807027 0.82823616 -0.46507642 0.7023594 -0.5075266 -0.56336015 -0.4397165 0.49094844 -0.3903013 0.46620616 -0.7257518 0.18183966 0.5173308 0.53875166 -0.8829515 0.27861157 0.52437663 -0.6927579 -0.57168883 0.63886696 0.5708137 -0.3998206 -0.73399425 -0.7485943 0.3610033 -0.41753525 -0.774967 -0.49759004 0.69661546 -0.32357845 -0.7685163 0.49103954 0.8322673 -0.54382855 -0.7889786 -0.5515944 -0.7419407 0.5151147 0.6659636 0.8105721 -0.62086254 -0.68138576 0.6019383 0.49395636 -0.5919031; -0.67097485 -0.614678 0.49487814 -0.75132066 -0.78599983 0.6940516 0.6739719 -0.6302515 0.28183544 -0.6553605 0.6151874 0.44663638 -0.5526022 0.76032156 0.5189642 0.7111126 -0.63280827 0.7503649 -0.46122026 0.6930808 0.46929476 -0.5498066 -0.37203455 0.7093866 -0.7034305 -0.7390851 -0.3899539 0.6910522 -0.50926995 -0.7271072 0.677889 0.49411637 -0.5934422 0.51338667 -0.785729 0.4817192 -0.67772436 0.49820298 -0.5813812 0.6998746 0.6585944 0.5001477 -0.543413 0.7645313 -0.56842 0.599158 -0.6210822 -0.6338545 0.54356676 0.6497077 0.6104468 -0.58837014 0.82298523 0.77202463 -0.74416095 -0.5837193 0.4576205 0.38883987 -0.66923434 -0.4985664 0.43457732 0.61565435 0.6665672 -0.759213 0.3526083 -0.51012534 -0.38394955 -0.61399126 0.6650105 0.74815637 0.6737149 0.5601267 -0.5565429 0.5072223 -0.3604723 0.49087253 0.7265022 -0.69778764 -0.46261573 0.7633463; -0.6379053 -0.77867025 0.5801185 -0.34640718 -0.44981065 0.59141517 0.6522039 -0.67364216 -0.94570035 -0.7172353 0.84533995 0.7373026 -0.89063084 0.46236756 0.9159542 0.29954883 -0.51756245 0.43647292 -0.7961638 0.57631284 0.6113019 -0.9021186 -0.69906205 0.48973608 -0.7729404 -0.4333652 -0.61526376 0.5477475 -0.6017421 -0.5695937 0.4257471 0.5257491 -0.8718677 0.58147305 -0.62367946 0.308532 -0.9376188 0.5415861 -0.46197304 0.302927 0.33640364 0.3803881 -0.838051 0.2771926 -0.65649647 0.35468322 -0.8117964 -0.60427135 0.3023329 0.68044907 0.7582238 -0.6192964 0.5540741 0.43233404 -0.65461415 -0.7436949 0.30174524 0.7247801 -0.8322856 -0.5694897 0.38612854 0.5746208 0.55347234 -0.8989986 0.7922799 0.71563405 -0.7324093 -0.7637805 0.527736 0.8114247 0.4055926 0.6696257 -0.36664933 -0.19944343 -0.6190415 0.6123559 0.5683483 -0.486345 -0.5752664 0.8033879; 0.7371509 0.4207784 -0.5622471 0.51182896 0.655585 -0.5316637 -0.5923509 0.7316254 0.2116484 0.7227703 -0.6036222 -0.7270902 0.3540048 -0.67166513 -0.64466274 -0.7946836 0.8540823 -0.6398886 0.4401468 -0.73603576 -0.73318696 0.48496225 0.5577268 -0.5996469 0.5915944 0.58878756 0.6294145 -0.5762964 0.6607639 0.44947395 -0.5471224 -0.5360432 0.57897097 -0.69838816 0.6514372 -0.650874 0.39889604 -0.5926907 0.7343137 -0.6355103 -0.6486965 -0.5301727 0.34692544 -0.6063376 0.6364963 -0.5947934 0.3847761 0.8967922 -0.61592704 -0.6445922 -0.4274802 0.807743 -0.6043559 -0.85568845 0.40786347 0.4631481 -0.6946341 -0.363885 0.5126034 0.61634535 -0.5907265 -0.87247163 -0.764291 0.32882765 -0.3266803 -0.43824446 0.66401 0.32113078 -0.6420892 -0.67499065 -0.53105956 -0.5149011 0.6568716 1.1946776 0.5196685 -0.9190563 -0.40331337 0.6740165 0.8657769 -0.6675951; 0.57852906 0.6805928 -0.7331 0.5454564 0.49177045 -0.46942094 0.7267351 0.6593363 0.58018696 0.4296998 -0.70295036 -0.6771751 0.38811654 -0.72351635 -0.5595939 0.0765846 0.7712249 -0.57184446 0.57487994 -0.51704043 -0.4722845 0.68633497 0.6105996 -1.0859526 0.70067835 0.6542237 0.7054078 -0.8590797 0.67798513 0.4846729 -0.70074517 -0.8475671 0.59263426 -0.60408103 0.62556 -0.73752034 0.4277887 -0.7055674 0.562192 -0.5821569 -0.499952 -0.6727093 0.7431364 -0.49408066 0.69130915 -0.8479695 0.6588449 0.6808882 -1.0797585 -0.5894577 -0.5331291 0.69227946 -0.67668164 -0.5223824 0.58185893 0.5385467 -0.8103085 -0.7001921 -0.03448739 0.8140942 -0.5134558 -0.6609264 -0.4585312 0.56656593 -0.13716522 -0.61975724 0.5637702 0.6512667 -0.75540537 -0.73882604 -0.57708347 -0.6894741 0.6410858 0.57719684 0.6411478 -0.7398903 -0.6768509 0.6303312 0.5997153 -0.63861614; 0.5720335 0.75414664 -0.45430204 0.6280203 0.5448342 -0.6082189 -0.59080064 0.5037612 0.65379417 0.57103264 -0.6343506 -0.46707976 0.55460405 -0.75551593 -0.6324929 -0.6914308 0.45439714 -0.5247346 0.7522325 -0.53923106 -0.76271164 0.79294145 0.6317906 -0.6245666 0.61390203 0.77244556 0.6967993 -0.49547467 0.53693384 0.46654937 -0.7278477 -0.6596806 0.48680532 -0.46394077 0.7424398 -0.45697004 0.711205 -0.6739286 0.48885068 -0.40454006 -0.52135605 -0.6638572 0.568454 -0.52645355 0.75907177 -0.4161241 0.68006545 0.38822013 -0.4647519 -0.6689401 -0.6237001 0.618041 -0.63422835 -0.42499277 0.71702236 0.57984275 -0.4935054 -0.56206185 0.78957725 0.4696951 -0.68162125 -0.4266811 -0.5743615 0.43839523 -0.7435648 -0.90812975 0.6752088 0.52035743 -0.75246096 -0.7717826 -0.48440683 -0.4513122 0.67557776 -0.94837856 0.87169576 -0.49862248 -0.68917775 0.6843531 0.5609023 -0.47951722; 0.71416634 0.50542253 -0.6231101 0.6529123 0.67046607 -0.67350614 -0.7813845 0.46363714 0.61843747 0.63402456 -0.3864075 -0.49828348 0.61097234 -0.7163996 -0.42070618 -0.52714115 0.687976 -0.80101603 0.73500365 -0.66996807 -0.6665151 0.6067172 0.7247418 -0.8239791 0.5073116 0.5061512 0.57978344 -0.65508616 0.6031796 0.5071781 -0.6065838 -0.81726164 0.42275858 -0.80047596 0.6260398 -0.80486214 0.5392116 -0.51198405 0.5367989 -0.46138567 -0.51464975 -0.46813902 0.7292248 -0.6407307 0.5848959 -0.6529367 0.7156498 0.50383455 -0.4502028 -0.7229701 -0.3920288 0.6567753 -0.54434216 -0.77388823 0.5867652 0.6854542 -0.7067084 -0.44308838 0.68056166 0.5863343 -0.8016927 -0.6006781 -0.61729914 0.5544315 -0.28727227 -0.73062944 0.46102497 0.4199146 -0.60883284 -0.7414312 -0.47931585 -0.6157561 0.4862793 0.6386675 0.65565956 -0.60814226 -0.52360207 0.755189 0.54696 -0.73425215; -0.5810106 -0.46913105 0.7160813 -0.5007349 -0.6636535 0.29199305 0.5118021 -0.35566396 0.5507456 -0.5095055 0.56037724 0.5199137 -0.50846535 0.57297534 0.3678646 0.5832674 -0.6140114 0.43727863 -0.4093706 0.7818394 0.62889045 -0.53624976 -0.5154492 0.58410144 -0.68026173 -0.77572286 -0.52259827 0.60660446 -0.49824134 -0.78806156 0.7840948 0.69764966 -0.37195912 0.7771248 -0.56369966 0.7818865 -0.569186 0.8053927 -0.5797151 0.5072119 0.42555517 0.5258816 -0.593134 0.7108739 -0.4579029 0.72731924 -0.41026255 -0.511814 0.75626224 0.6663614 0.45061916 -0.47819996 0.4386027 0.57520056 -0.6126463 -0.55134773 0.61897224 0.48874855 -0.52229565 -0.5377826 0.55202746 0.44641116 0.5120972 -0.24023837 0.015649619 -0.7737545 -0.45009926 -0.24454899 0.54119575 0.60515225 0.56169724 0.36967102 -0.64449656 0.40814978 0.72172 0.63553065 0.62628525 -0.7179449 -0.8435516 0.6548658; 0.5196046 0.5518133 -0.43154174 0.72012836 0.5126749 -0.6606117 -0.36806706 0.4238665 0.7488043 0.5153586 -0.72414964 -0.76558566 0.45860597 -0.5610776 -0.6502375 -0.71369743 0.8050656 -0.58492684 0.67232114 -0.4801216 -0.54503936 0.68865204 0.5714254 -0.8002693 0.4700952 0.62406135 0.7891171 -0.7919975 0.61702675 0.65095705 -0.4870683 -0.50986594 0.38466796 -0.5272951 0.7468492 -0.78059316 0.71803445 -0.5812123 0.42030078 -0.49286526 -0.8021714 -0.5810451 0.45763564 -0.49479902 0.44186983 -0.49679622 0.62796223 0.4912285 -0.58546424 -0.4295381 -0.67548734 0.5028181 -0.43958664 -0.46884552 0.4279698 0.5208819 -0.7056742 -0.77975464 0.4505409 0.48845607 -0.4868714 -0.7971122 -0.68043846 0.42014185 -0.5000352 -0.71474874 0.622845 0.5446351 -0.5611391 -0.7743326 -0.434625 -0.6452496 0.6340911 0.6761672 0.68082696 -0.4768785 -0.7566558 0.45101717 0.63497174 -0.78122914; -0.47048378 -0.6901948 0.7733985 -0.65910524 -0.67633003 0.73135066 0.77755916 -0.63061553 -0.4115871 -0.60398054 0.5705253 0.54990685 -0.4768537 0.55660075 0.6603379 0.40236953 -0.6345822 0.545914 -0.42583215 0.41319638 0.66772217 -0.8234594 -0.6976185 0.43117145 -0.42960733 -0.4712826 -0.45100424 0.5191301 -0.4664068 -0.6832072 0.7000936 0.49371004 -0.5264289 0.5103186 -0.4789802 0.6126467 -0.82992697 0.4674527 -0.53120464 0.5760203 0.5572774 0.64481276 -0.7837273 0.42522058 -0.7504663 0.6982224 -0.6879463 -0.28706688 0.57059216 0.6460113 0.8245538 -0.39949724 0.51134115 0.50010073 -0.6900972 -0.8032069 0.5833964 0.70329946 -0.65862405 -0.7575164 0.5829426 0.5558573 0.6574712 -0.8938443 0.69621193 0.07775893 -0.48863927 -0.6445061 0.53053015 0.5217152 0.49818832 0.6974304 -0.7426538 0.74220884 -0.8378785 0.5627692 0.5490174 -0.55156934 -0.61176074 0.55825275; 0.75613046 0.5226329 -0.4617435 0.59493417 0.7928731 -0.6255165 -0.65925676 0.5586062 -0.36233312 0.57964075 -0.7487485 -0.4132347 0.5458708 -0.6014974 -0.603102 -0.68660355 0.453538 -0.48682752 0.62697494 -0.7209437 -0.68269354 0.4826338 0.6761115 -0.41690236 0.47082707 0.47960892 0.48984468 -0.44529316 0.61525756 0.6028971 -0.56140935 -0.78427285 0.5011224 -0.6540006 0.7389193 -0.4768456 0.6741569 -0.59017247 0.5879723 -0.6440556 -0.5567007 -0.5598618 0.6649435 -0.5309465 0.53003025 -0.46484908 0.5700428 0.22673486 -0.5756854 -0.39797738 -0.50157005 0.5968865 -0.63431257 -0.5708741 0.72270834 0.50711143 -0.60387003 -0.47072077 0.73197496 0.7267835 -0.6561605 -0.54476565 -0.77218616 0.73547 -0.64838046 0.6339578 0.5748177 0.7171061 -0.5760144 -0.7563857 -0.4636869 -0.75040287 0.634311 -0.38559994 -0.7115055 -0.4750713 -0.6405313 0.53473634 0.5320512 -0.49551743; -0.78093004 -0.4065913 0.65691453 -0.57394254 -0.68561107 0.50496984 -0.47251055 -0.76604944 -0.78218806 -0.6179005 0.4890211 0.67707074 -0.563926 0.6032055 0.73170227 -0.71522707 -0.4371632 0.78595304 -0.4881667 0.5880596 0.72439057 -0.5656378 -0.49451315 0.31467 -0.63891786 -0.48522016 -0.68497473 0.64853185 -0.475676 -0.61849916 0.46159333 0.5920275 -0.4029389 0.69098616 -0.47843543 0.62930757 -0.7648202 0.6025227 -0.45809093 0.66475165 0.5063782 0.44558427 -0.53941536 0.6628399 -0.58706206 0.5070318 -0.5386022 -0.6464351 0.35683686 0.662598 0.51689106 -0.75995517 0.5527574 0.43949687 -0.5139457 -0.6511277 0.59575814 0.44397187 -0.73023343 -0.41578823 0.48675397 0.68831235 0.46481073 -0.65569097 0.59797597 0.7624802 -0.6779097 -0.58847 0.62401867 0.62097645 0.43418673 0.77439165 -0.43748856 -0.6212503 -0.7743711 0.79843104 0.63851947 -0.5201358 -0.43917936 0.5631595; -0.41992947 -0.5818348 0.64872193 -0.6601149 -0.58440214 0.7583224 0.74109787 -0.7577633 -0.5131068 -0.4239021 0.44323123 0.62524796 -0.38447928 0.46192825 0.6969642 0.48125905 -0.78749704 0.46332777 -0.7286258 0.5979113 0.4753292 -0.6322305 -0.5989672 0.7558226 -0.5959515 -0.3163171 -0.45975015 0.5514584 -0.6713217 -0.48799202 0.48856556 0.96789074 -0.68005174 0.7457904 -0.52303207 0.48048067 -0.7313268 0.75733864 -0.6982506 0.50767744 0.79620063 0.78672534 -0.72702706 0.70707744 -0.48240614 0.709881 -0.5254198 -0.438178 0.44615743 0.69197786 0.60985315 -0.46441075 0.8047839 0.561412 -0.7246412 -0.41933212 0.7309144 0.4813414 -0.6061372 -0.6903926 0.8213269 0.58864546 0.77738065 -0.3968663 0.657434 0.4962853 -0.72392184 -0.74472827 0.5674302 0.5261801 0.58097386 0.50146353 -0.67214787 -0.4968542 -0.5185831 0.70046365 0.5872917 -0.5691974 -0.6094397 0.444528; -0.6403352 -0.3887312 0.5096133 -0.7359007 -0.70632166 0.75971985 -0.56784534 -0.550348 -0.4304427 -0.73588026 0.62145096 0.6392672 -0.704497 0.54831415 0.70741165 1.5675291 -0.55804527 0.70000684 -0.7807235 0.7154125 0.49753106 -0.62584794 -0.724261 0.56769717 -0.5886159 -0.45318827 -0.5812001 0.56800735 -0.41184208 -0.44416642 0.7227029 0.5034454 -0.45669037 0.49188027 -0.5681497 0.6038994 -0.6743139 0.6626402 -0.59233123 0.5163673 0.71045935 0.61876607 -0.6209411 0.601868 -0.74267375 0.8098275 -0.4801861 -0.6226012 0.64330494 0.4051768 0.6769107 -0.7007909 0.4739577 0.4698341 -0.52872926 -0.6869296 0.8290043 0.729155 -0.5173259 -0.49610895 0.5960384 0.6026558 0.5059125 -0.6019953 0.40846354 0.5823874 -0.53487986 -0.4552199 0.77684015 0.7373974 0.71556425 0.73950535 -0.7919716 -0.66789436 -0.70018834 0.5353115 0.76804787 -0.48809832 -0.49102998 0.7776907; -0.50355864 -0.7102732 0.4990041 -0.4820559 -0.46075878 0.42778087 0.6496515 -0.48849595 0.44544742 -0.71856517 0.7362626 0.71917135 -0.55009705 0.808375 0.65022147 0.7040015 -0.64056623 0.67712754 -0.6525521 0.5405565 0.79633176 -0.46038496 -0.56040406 0.50175667 -0.60172486 -0.61585736 -0.43956074 0.519938 -0.6506524 -0.60638493 0.7092349 0.6110919 -0.43435183 0.5778088 -0.6995057 0.53811896 -0.47366294 0.7283035 -0.49607828 0.6561498 0.498162 0.7840938 -0.5774228 0.5848017 -0.80427474 0.67575455 -0.7520857 0.77501804 0.8013046 0.6783739 0.48699862 -0.7120317 0.48511052 0.5165868 -0.7034196 -0.72203934 0.47671595 0.5373463 -0.42911014 -0.45828545 0.46595764 0.5699745 0.48548892 -0.7247714 0.026488438 -0.70243394 -0.52611595 -0.4788719 0.6842543 0.59092736 0.5546138 0.6044318 -0.53747064 0.4698866 0.6858265 0.53535885 0.7229834 -0.5235625 -0.14101951 0.7232077; -0.57721305 -0.760934 0.57561296 -0.58634144 -0.56798875 0.709621 -0.62483823 -0.8250737 -0.62003106 -0.6088915 0.6570802 0.84717864 -0.6957627 0.5748603 0.6027294 -0.58065826 -0.70974505 0.5669555 -0.74927497 0.6580652 0.7011725 -0.49134314 -0.725097 -0.40763944 -0.69539493 -0.71137565 -0.7833957 0.576703 -0.759323 -0.5973813 0.6285497 0.037142735 -0.72447646 0.46492174 -0.5600276 0.21733078 -0.50861716 0.35940844 -0.5302115 0.73920256 0.44679517 0.6300965 -0.5023567 0.61628217 -0.7072954 0.41776004 -0.68818146 -0.4729576 -0.56877714 0.7829966 -0.5132402 -0.74555516 0.43108296 0.5286484 -0.75900996 -0.6020601 0.30817145 0.6931785 0.3953703 -0.63115156 0.44992608 0.6366372 0.66196245 -0.66282135 0.62580216 0.5678797 -0.83058643 -0.58722174 0.5443426 0.56296694 0.57440794 0.59858906 -0.6176631 -0.53511506 -0.57372063 0.60739964 0.80291337 -0.61594117 -0.39153877 0.6532386; 0.2232326 0.123601794 0.73507833 -0.7079807 -0.6150508 0.6744879 -0.48205018 0.3726854 -0.78078645 -0.94330776 0.63288206 -0.20179273 0.3508338 0.61863965 0.51130474 -0.50408256 -0.38782036 0.49486515 0.088006504 0.6945372 0.6476128 0.33810082 -0.7191051 -0.80407596 -0.080944024 -0.567811 0.22237939 -0.7650806 -0.80227494 0.6056433 -0.72933906 -0.65508056 0.61440194 0.5820918 -0.5636045 -0.629792 -0.5104006 -0.5026747 -0.47270793 -0.22702819 0.6204757 0.78094757 -0.7499856 0.47695094 -0.7343104 -0.17017888 0.6005848 -0.53973883 -0.5918951 -0.2941323 -0.6160548 -0.5745859 0.26363152 0.76476777 0.23721652 -0.6233145 -0.5639633 -0.5316395 0.6419099 -0.6019462 0.40530252 0.5373401 -0.3730875 -0.59665984 0.19760475 0.7701875 -0.6708038 -0.5887165 -0.60042346 -0.016762441 0.5257645 0.5798801 0.6100035 -0.469323 -0.54228485 0.17296322 0.8181592 -0.7182878 -0.62351006 0.8130135; -0.6322024 -0.6675203 0.47807416 -0.7678223 -0.80131024 0.53352165 0.69634855 -0.7316206 -0.49786177 -0.771143 0.52149224 0.6509672 -0.7142209 0.66107726 0.62136394 0.79356927 -0.6762371 0.44369242 -0.6678204 0.6941088 0.73541266 -0.6525186 -0.6416992 0.47693437 -0.7969941 -0.49095052 -0.54484105 0.49387723 -0.791471 -0.5126626 0.79947567 0.6283864 -0.40424648 0.48257515 -0.4846972 0.459095 -0.40590712 0.55907285 -0.71021134 0.71483856 0.6323035 0.5022259 -0.6302107 0.47648638 -0.5271035 0.6520191 -0.63435805 -0.5671744 0.8337451 0.629171 0.42681292 -0.7680278 0.48453292 0.6213284 -0.68004745 -0.76417816 0.48043367 0.7960191 -0.407316 -0.70731395 0.5367606 0.6059766 0.57336056 -0.59162855 0.19307995 0.57389444 -0.69640243 -0.49187776 0.7596982 0.5085671 0.47781578 0.61680764 -0.5501352 -0.6750961 -0.36381724 0.7775482 0.65090764 -0.49412814 -0.7441985 0.5574252; -0.7577632 -0.60047984 0.44551164 -0.6667109 -0.5969815 0.57497823 0.76743567 -0.60249716 -0.7012355 -0.49163842 0.4122845 0.4495131 -0.63088363 0.60594934 0.44898814 0.64621794 -0.71123356 0.65342444 -0.6017415 0.5380041 0.70971155 -0.43188086 -0.4155169 0.5176749 -0.6662582 -0.5680814 -0.74748796 0.6496957 -0.5772506 -0.74792033 0.54909205 0.71521986 -0.74444574 0.5131168 -0.48561394 0.45551696 -0.4128195 0.49240682 -0.5720433 0.5291395 0.48866028 0.73842365 -0.6851238 0.5475637 -0.41584143 0.4582836 -0.65316105 -0.460396 0.6968582 0.5082371 0.69873875 -0.61123335 0.5796752 0.72286654 -0.5673212 -0.77504003 0.41311112 0.66902715 -0.6296823 -0.5355998 0.5645133 0.73809934 0.6354469 -0.6796483 0.6976171 0.50458986 -0.7318461 -0.64246583 0.7815981 0.6393963 0.41873044 0.71860975 -0.51518655 0.5233012 -0.76412255 0.4144556 0.6203793 -0.41355914 -0.49265483 0.62112075; 0.65404767 0.65850127 -0.68492275 0.5499669 0.63825274 -0.8724715 -0.57832485 0.46847814 -0.06887414 0.7509661 -0.60439014 -0.83160704 0.8559205 -0.5119853 -0.80501246 -0.50324935 0.46813837 -0.75163454 0.74934477 -0.6740122 -0.66929644 0.511211 0.5618662 -0.49404392 0.613585 0.45990512 0.6294121 -0.4089011 0.53281224 0.5643243 -0.3861264 -0.43585876 0.5351997 -0.63892496 0.5061945 -0.6883251 0.8508592 -0.44615024 0.73912174 -0.43924326 -0.69553876 -0.7769412 0.4721553 -0.42696196 0.65840346 -0.5118116 0.8482628 0.017603641 -0.50155807 -0.594463 -0.80061364 0.3631404 -0.4303955 -0.7132045 0.81377774 0.76792145 -0.25230214 -0.69288796 0.54547644 0.7266001 -0.3952585 -0.7710136 -0.59901786 0.6676147 -0.6399112 0.51975334 0.8284598 0.6495592 -0.38091314 -0.56747216 -0.52646476 -0.50033987 0.55117214 -0.66851574 -0.25194418 -0.40729505 -0.80794466 0.6586695 0.033812344 -0.6198042; -0.55514324 -0.67003137 0.50312436 -0.70718396 -0.59923965 0.43164235 -1.068438 -0.4633155 -0.5625414 -0.47374654 0.66673607 0.45028743 -0.67296463 0.4629077 0.73806906 0.4899407 -0.6247185 0.54541266 -0.44161338 0.47265115 0.64662963 -0.60808295 -0.44425923 0.74649286 -0.5444476 -0.51379526 -0.59280336 0.7724193 -0.6371531 -0.6582351 0.6757827 0.7472298 -0.67837846 0.4282699 -0.6349842 0.5475856 -0.458342 0.55661905 -0.68975127 0.45305035 0.77787256 0.4927064 -0.4489513 0.43043783 -0.71082824 0.48657247 -0.5343794 -0.753431 0.63728106 0.44384983 0.6853241 -0.7178618 0.42331296 0.5878205 -0.4945395 -0.550434 0.7361459 0.77218854 -0.5837385 -0.53593165 0.60741526 0.65921515 0.60348994 -0.4756333 0.6861137 0.679074 -0.7198523 -0.7481716 0.6890557 0.50176394 0.71138185 0.7430428 -0.5394884 -0.6317822 -0.72873175 0.6108311 0.633059 -0.54492414 -0.4972839 0.63276863; 0.40526918 0.6066615 -0.55172133 0.6024778 0.49279636 -0.484247 0.76436424 0.70842385 0.3976546 0.57308686 -0.38948402 -0.4000011 0.6636343 -0.6464143 -0.59022087 0.3769261 0.6717987 -0.72000223 0.3807061 -0.6120071 -0.38648608 0.6646544 0.7827524 -0.7800983 0.5359362 0.5817262 0.5419954 -0.80990887 0.5391047 0.671781 -0.78723407 -0.764772 0.81427085 -0.4772576 0.62714565 -0.42361078 0.7285618 -0.5620575 0.7504872 -0.4311223 -0.51703924 -0.61109704 0.7465402 -0.50117123 0.6270353 -0.78567725 0.46001926 0.6250845 -0.5039758 -0.41963473 -0.45077357 0.51249546 -0.43115544 -0.5663501 0.55675715 0.370143 -0.82166386 -0.70642555 0.81120497 0.7235334 -0.685443 -0.6400034 -0.6223529 0.3726016 -0.671098 -0.46278092 0.40852728 0.6756323 -0.5009401 -0.5761854 -0.42784908 -0.5095178 0.5248227 0.7860396 0.534838 -0.5667381 -0.4559012 0.5204999 0.5455491 -0.38970172; 0.8286542 0.49548602 -0.7815278 0.6049922 0.7674566 -0.45695382 -0.65839446 0.6834321 0.6695459 0.59262913 -0.4684206 -0.46170548 0.7417332 -0.47113034 -0.63012004 -0.68893766 0.5363964 -0.5408783 0.6866059 -0.58630306 -0.6039671 0.6453846 0.61954767 -0.47159648 0.6553067 0.6148086 0.3239768 -0.5056022 0.610715 0.5132403 -0.6170926 -1.0033259 0.73747057 -0.697003 0.42807177 -0.5511981 0.77518016 -0.52894497 0.64044136 -0.69113064 -0.61822414 -0.60872763 0.603347 -0.5903583 0.53880477 -0.45795646 0.63795376 0.6503552 -0.5598092 -0.62030274 -0.56075263 0.5583789 -0.79901224 -0.75260794 0.63400054 0.4817989 -0.40705556 -0.736878 0.49235526 0.66143554 -0.47846523 -0.6721554 -0.7604528 0.49428725 -0.71806866 -0.7608461 0.66326654 0.7451062 -0.45286798 -0.4247628 -0.6159409 -0.47072184 0.4925408 0.16256483 0.5467135 -0.50853926 -0.44020826 0.54782486 0.6349657 -0.57942176; -0.758815 -0.4609818 0.5341296 -0.6769586 -0.6848903 -0.7328767 0.56860965 -0.6879721 0.7590056 -0.41065985 -0.4568105 0.50171185 -0.56826085 -0.81258667 -0.6644087 0.4455955 -0.6594478 0.37280574 -0.4283819 -0.535586 0.6364907 -0.4549963 0.47356272 0.7630639 -0.6972684 -0.6711351 -0.5548957 0.77865756 -0.40092096 -0.5608697 0.73284274 0.6123697 -0.6601931 0.41831735 -0.57954127 0.40571836 0.40974933 0.7499471 0.8283353 0.47713375 0.4647388 0.4411237 -0.4712308 0.27367252 0.6842296 0.5940051 -0.5311444 0.6152405 0.72402793 0.52117133 0.55737907 0.7204638 0.7216347 -0.46166646 -0.5585029 -0.4975946 0.6333427 0.60163563 -0.6321711 0.78634006 -0.6144781 0.74085397 0.4864469 0.45941246 -0.5999723 -0.60078436 -0.4995374 0.33854747 0.6011946 0.65255946 0.6237465 0.6554458 -0.6704981 0.697985 0.43352205 0.48764113 0.69501364 -0.34961572 0.58984935 0.44818458; -0.7807861 -0.75966924 0.6921889 -0.71586865 -0.5603338 0.70217574 0.6054849 -0.7055523 0.21924646 -0.63889354 0.69203013 0.82628286 -0.63960576 0.5289392 0.6391439 0.4885664 -0.47000542 0.55617154 -0.64632857 0.41057992 0.5855433 -0.6074674 -0.6105536 0.5081207 -0.58693475 -0.49705368 -0.6608147 0.5408484 -0.75761557 -0.61620545 0.5650532 0.5352307 -0.46566972 0.6693307 -0.74643654 0.46786514 -0.78724277 0.391853 -0.38681564 0.75367624 0.57876474 0.49041525 -0.50456524 0.69898593 -0.4541097 0.4843182 -0.5218161 1.2483461 0.589327 0.5301396 0.5961406 -0.48927343 0.58996576 0.5035988 -0.6102765 -0.5219744 0.38895822 0.726847 -0.48561904 -0.536622 0.51039773 0.712822 0.5232684 -0.70889556 0.4932381 -0.49483734 -0.5489701 -0.86542827 0.61376065 0.62144125 0.74910647 0.6082719 -0.64805776 0.860168 0.3520183 0.41171387 0.59709525 -0.677838 0.15521151 0.71198475; 0.5022936 0.52883935 -0.59167176 0.50997955 0.58594173 -0.4119715 0.85336804 0.7472527 0.35546637 0.6635789 -0.39018413 -0.506485 0.3821834 -0.53828126 -0.4375565 0.31100214 0.63200825 -0.570718 0.44686478 -0.52718604 -0.43596962 0.68947303 0.77088994 -0.84708035 0.4651396 0.43659073 0.5851704 -0.719159 0.51929444 0.7017306 -0.7700261 -0.53098184 0.76931417 -0.77519304 0.52580494 -0.7138783 0.5923972 -0.6980252 0.4992382 -0.6412066 -0.60064596 -0.7408092 0.47038576 -0.65791816 0.68508315 -0.7842387 0.53163505 0.6162098 -0.5178234 -0.5784797 -0.38317624 0.41746414 -0.51567066 -0.7488468 0.73894906 0.66849834 -0.7178888 -0.71488214 0.6511471 0.46759444 -0.5073931 -0.48114246 -0.76625305 0.5499009 -0.61639214 -0.73394156 0.40683994 0.56672806 -0.72143835 -0.66417474 -0.7097241 -0.4888201 0.46882123 0.62538105 0.7136873 -0.4866863 -0.6548586 0.4245946 0.67145675 -0.43849963; 0.44957623 0.6830803 -0.62644696 0.723811 0.5440976 -0.5761873 -0.749742 0.41502267 0.5428809 0.6355626 -0.7302607 -0.62922066 0.63021195 -0.43674007 -0.46889573 -0.3657079 0.4026362 -0.49540076 0.7412409 -0.67133206 -0.72455955 0.53479654 0.49259683 -0.3990173 0.6373179 0.4729027 0.48213306 -0.60230476 0.7233402 0.43891677 -0.51624274 -0.73367214 0.8122038 -0.563545 0.48410884 -0.632952 0.7795789 -0.5587034 0.49217966 -0.7386571 -0.54122955 -0.64348817 0.75038147 -0.4234007 0.47469932 -0.5028326 0.6448594 0.5013286 -0.6922278 -0.7148285 -0.5077307 0.45034325 -0.7191498 -0.5825384 0.44695318 0.6541794 -0.5098252 -0.5686911 0.56765467 0.407693 -0.50080603 -0.5332867 -0.57358354 0.7727047 -0.66412324 -0.6142154 0.55422497 0.7855209 -0.6728 -0.43166196 -0.5241972 -0.65427434 0.49065086 -1.5117362 0.79949033 -0.5674359 -0.5988518 0.51277524 0.61729294 -0.64675033; 0.4798252 0.65450174 -0.7656371 0.5770405 0.6414657 -0.67588156 0.87025034 0.66293377 0.4649678 0.64864886 -0.73318964 -0.5867529 0.607854 -0.74214387 -0.71033865 -0.5536056 0.576007 -0.5634444 0.46606633 -0.78302497 -0.64192486 0.613092 0.6281448 -0.55825675 0.40885484 0.46984148 0.6655484 -0.49834654 0.6641307 0.5138687 -0.7783618 -0.59635115 0.4404811 -0.44878206 0.5106107 -0.55036896 0.6865197 -0.4592986 0.5498722 -0.5853562 -0.6602839 -0.4388204 0.51454484 -0.76012444 0.6484605 -0.7313198 0.47100517 0.46444196 -0.72425526 -0.69348377 -0.39912602 0.4523965 -0.5650201 -0.4777663 0.6792954 0.60686296 -0.58365935 -0.5918872 0.4965158 0.5967812 -0.68363357 -0.6081666 -0.69677454 0.750273 -0.6443768 -0.42381606 0.6664541 0.6671114 -0.79309 -0.4287672 -0.46910742 -0.6642191 0.5054133 0.5464569 0.6025514 -0.5491449 -0.38542956 0.4486669 0.5623951 -0.72675604; 0.52396727 0.53107303 -0.6242652 0.65621 0.43282855 -0.8594254 -0.5494451 0.659848 0.06371163 0.42903253 -0.79303837 -0.6436434 0.5224542 -0.79521054 -0.49377847 -0.6765265 0.5030154 -0.62285244 0.6034117 -0.7272989 -0.5626344 0.49969262 0.55338025 -0.721195 0.6805784 0.49576014 0.65457225 -0.7260361 0.5986939 0.67921704 -0.4935655 -0.8085183 0.6224113 -0.67090696 0.6197496 -0.5285511 0.63230425 -0.6754446 0.42761263 -0.56008387 -0.4596573 -0.6988245 0.5119927 -0.6582758 0.4226629 -0.47224995 0.4970321 0.48755863 -0.581003 -0.61314094 -0.4591757 0.43416163 -0.76440954 -0.74998796 0.62790984 0.7380407 -0.311672 -0.5887329 0.53473085 0.6160466 -0.47366327 -0.6218962 -0.46389118 0.86748815 -0.8749996 0.39008805 0.5872 0.8374851 -0.5055976 -0.67133594 -0.4496224 -0.38981465 0.6346926 -0.95124537 0.29288292 -0.54273593 -0.7330948 0.6321965 0.592744 -0.6897672; -0.69756174 -0.5640565 0.764606 -0.58656555 -0.50642604 -0.43114161 0.7821527 -0.5428118 0.478807 -0.687766 0.50266993 0.4962668 -0.65406334 -0.6258855 0.7097166 0.61473876 -0.64752126 0.71388245 -0.55781174 -0.3841773 0.69953763 -0.53714633 0.64251214 0.68574286 -0.46590498 -0.6033569 -0.61326116 0.66152817 -0.41818613 -0.46959895 0.7380913 0.55181307 -0.5469671 0.6158624 -0.7773022 0.5076838 0.66216993 0.653571 0.3255486 0.5137656 0.5064429 0.5824011 -0.56348664 0.6938642 -0.49631238 0.559873 -0.49561656 0.4532106 0.72059286 0.48956412 0.74070823 -0.6746275 0.45307347 -0.58353573 -0.586201 -0.43779463 0.65506387 0.5914652 -0.42860034 -0.47902358 -0.44578832 0.6929664 0.7233609 0.49952158 -0.5474708 -0.4460412 -0.70554477 -0.35096437 0.5742424 0.67153996 0.58158493 0.65723616 -0.5590152 0.49388546 0.46368417 0.45695257 0.7624424 -0.66179156 0.70910215 0.68843955; -0.62468654 -0.6841838 0.7092269 -0.7132708 -0.70901453 0.67526233 0.56091505 -0.7007554 0.45403072 -0.6373054 0.5583477 0.5929611 -0.60346764 0.53108925 0.5136499 0.7022636 -0.56384456 0.43611133 -0.6141556 0.59607667 0.5613005 -0.5887006 -0.6817295 0.60086024 -0.7560585 -0.6012681 -0.71529275 0.68193066 -0.6123643 -0.51552 0.4251916 0.64442736 -0.5452623 0.60261065 -0.75423586 0.540529 -0.7280691 0.516149 -0.48093393 0.57883996 0.5846997 0.513874 -0.69779456 0.5998913 -0.5107445 0.7786522 -0.54222906 -0.5829898 0.50935423 0.47568133 0.72547966 -0.5693873 0.6808363 0.59518963 -0.5314916 -0.61126804 0.5312255 0.39132977 -0.6799393 -0.7678527 0.58483213 0.66865134 0.5269163 -0.7857344 0.63291043 -0.6159021 -0.7413914 -0.71914804 0.6695771 0.61408585 0.5932061 0.5635677 -0.75202334 0.54002285 0.22806065 0.76560074 0.5667776 -0.51620066 -0.67477 0.75536394; 0.11738321 0.4679782 -0.83328027 0.8554453 0.67602605 -0.636326 0.54523325 -0.39343384 0.62407273 0.8112059 -0.3501972 -0.6966331 0.32277575 -0.7269631 -0.60606617 0.45104766 0.66006714 -0.8366032 0.5875861 -0.5984467 -0.7855087 0.05296102 0.5467275 0.40532118 0.4011863 0.38844502 0.62512594 0.47909898 0.7345028 0.21368486 -0.16053025 0.680189 0.43878826 -0.6522937 0.820892 0.34789175 0.5018267 0.300549 0.84986705 -0.51490957 -0.8480434 -0.65638006 0.51131725 -0.57974654 0.84314615 -0.7212432 -0.4889107 0.6687083 0.56202847 0.44322833 0.73113143 0.5954093 -0.6388374 -0.6683112 -0.6019069 0.73128754 -0.66444296 0.05472339 -0.5972872 0.83736324 -0.6429927 -0.7539645 -0.7374648 0.44015798 -0.15931189 -0.43924516 0.45189402 0.33887023 -0.4445381 -0.58932954 -0.8917161 -0.5811889 0.39948758 0.5441152 0.5973986 -0.7127824 -0.62526447 0.5528289 0.70036644 -0.42597717; -0.87185895 -0.8969273 0.35360458 -0.5500811 -0.6977024 0.5852913 -0.37288517 -0.7908439 -0.86079806 -0.653164 0.82429934 0.7788626 -0.5509469 0.40796518 0.62498057 -0.73818517 -0.62104154 0.42611703 -0.8139796 0.46398783 0.57345456 -0.5975331 -0.7155976 0.34947336 -0.7907993 -0.7580913 -0.76747364 0.33568743 -0.7689977 -0.42244944 0.42945147 0.4148628 -0.8060147 0.7140951 -0.5821034 0.50743 -0.91059375 0.3472457 -0.53566146 0.53049284 0.65780234 0.51742744 -0.70583326 0.61779565 -0.41058096 0.5409731 -0.81549364 -0.5732886 0.3783809 0.8586523 0.6845091 -0.518258 0.67715496 0.59929675 -0.7280482 -0.70215076 0.435631 0.84562397 -0.3235406 -0.39778545 0.5190648 0.5180588 0.568273 -0.91574603 0.4667666 0.76318866 -0.7187541 -0.5400634 0.582887 0.5414927 0.70647615 0.6211612 -0.69913566 -0.44937178 -0.7370275 0.35486916 0.8045159 -0.6610948 -0.28237543 0.8248185], bias = Float32[-0.6108908; -0.68583715; 0.075259864; -0.6992091; -0.6167876; 0.62165856; -0.6237554; -0.69486195; 0.59648144; 0.59395343; -0.6441067; -0.61073583; 0.59166116; 0.68272436; -0.59153616; 0.6065619; 0.6440461; 0.576896; 0.7316022; 0.71500105; 0.60520947; 0.4589702; -0.40886247; -0.54402727; -0.53699267; 0.71700066; 0.5879719; 0.62739164; 0.61978585; 0.5955463; -0.58907425; 0.6768419; 0.63722587; -0.50914556; -0.5247153; 0.62806904; -0.6429787; 0.5255902; 0.59790695; -0.5218423; -0.62628454; 0.5868023; 0.5359011; 0.5818107; -0.61813337; 0.43471384; -0.57308793; -0.5651008; 0.5370155; 0.77499574; -0.43766877; -0.5571877; -0.60640454; -0.57334876; 0.605494; -0.51805884; 0.6698879; -0.5970554; 0.5559924; 0.6352511; 0.56682694; 0.5547547; 0.61673236; 0.48949865; 0.5611715; 0.5896692; -0.60524577; 0.58517486; -0.61002856; -0.6900688; -0.44286892; 0.59864974; -0.6023534; -0.6103745; -0.5801679; -0.65663606; 0.31396526; 0.5595314; -0.41917625; 0.68436474]), layer_3 = (weight = Float32[-0.6006161 0.65691483 -0.73442024 -0.56397206 -0.42027414 0.5527158 -0.5601335 -0.7005654 0.45085752 0.4532828 -0.6484087 -0.49621937 0.34333572 -0.5180401 -0.46131748 0.47483552 0.6447048 0.5484862 -0.6329792 0.19897845 0.59227353 0.8471579 0.7552669 -0.46370307 -0.3843718 0.54603064 0.34701326 -0.6536222 0.38885632 0.59109896 -0.4461714 -0.9226847 -0.4851689 0.5217957 0.82340324 0.59464663 0.80581504 -0.66520125 0.43187743 -0.46273664 -0.57338524 0.47621608 0.4280769 0.45523006 -0.4709106 -0.67544955 0.8073015 0.58479697 -0.45596373 -0.87842053 0.6580692 0.790418 -0.49480984 0.69208217 0.6360355 -0.4584885 -0.51657665 -0.60829985 0.5135967 -0.85847163 -0.8463854 -0.7999055 -0.53590953 -0.47504458 -0.78947735 0.40713692 0.5317361 0.43133745 -0.53647345 0.6181231 0.70169693 -0.632631 -0.47656423 -0.48147792 -0.4637407 0.5488103 -0.29354644 -0.53038186 0.7033099 -0.43315533; -0.37005228 0.6850734 -0.8518241 -0.53895515 -0.6485997 0.50752366 -0.33826414 -0.32391304 0.50975806 0.5735435 -0.31065628 -0.36666316 0.58842844 -0.79367423 -0.34864855 -0.12206667 0.31599694 0.42604464 -0.74270076 0.6858452 0.45209637 -1.0307194 0.37876323 -0.5236052 -0.52410865 0.66716915 0.61257446 -0.70348865 0.63097775 0.6277996 -0.78945535 0.65208954 -0.86630476 0.81270885 0.3775434 0.43215463 0.73112255 -0.9160239 0.59696615 -0.44010252 -0.39653677 0.3196915 0.44595015 0.71016824 -0.5642351 -0.51515436 0.77290714 0.8223671 -0.81466854 0.1636743 -0.20556675 0.56111205 -0.38784677 0.80500346 0.42586705 -0.38606986 -0.92973894 -0.3652333 0.6361355 -0.8691722 -0.5925124 -1.1721601 -0.79668874 -0.9493758 -0.920904 0.3719377 0.86468595 0.5073009 -0.56675386 0.73968077 0.45319718 -0.87300843 -0.614546 -0.30334592 -0.3904528 0.8288809 0.05620783 -0.95933914 0.5013473 -0.8272294], bias = Float32[-0.57957697; -0.61696047]))
loss_neuralode(p_trained_highnoise)